// Dialog Example - dialogex.cpp

#include "c4d.h"
#include "c4d_symbols.h"
#include "dialogex.h"

/* ****************************************************************
   Definitions of the 'DialogExample' class derived from 'GeDialog'
   
   This is the class which creates the dialog box and handles all
   the user interaction with the plugin
*******************************************************************/
DialogExample::DialogExample(void)
{
	// this is the constructor for the dialog box class - we don't need to do anything here
}

DialogExample::~DialogExample(void)
{
	// destructor for the dialog class, but there is nothing to do in this case
}

Bool DialogExample::CreateLayout(void)
{
	// C4D calls this function when the dialog is to be displayed,
	// so it must be overridden to display the dialog as you want it

	Bool result = true;

	// call the base class function first
	GeDialog::CreateLayout();

	// create a resource object and initialise it to the plugin's path
	GeResource dlg_res;
	result = dlg_res.Init(GeGetPluginPath());
	result = LoadDialogResource(IDD_OBJECTEDITOR, &dlg_res, 0);

	// we have to set up the two combo boxes in the dialog, with IDs to indicate which option was chosen, and some text for the user
	// first the editor visibility
	FreeChildren(IDC_EDITORVIS);			// clear the combo first; if we don't do this C4D may add multiple menu entries
	AddChild(IDC_EDITORVIS, 6050, "Always visible");
	AddChild(IDC_EDITORVIS, 6051, "Always invisible");
	AddChild(IDC_EDITORVIS, 6052, "Depends on parent");

	// and now the render visibility
	FreeChildren(IDC_RENDERVIS);
	AddChild(IDC_RENDERVIS, 6060, "Always visible");
	AddChild(IDC_RENDERVIS, 6061, "Always invisible");
	AddChild(IDC_RENDERVIS, 6062, "Depends on parent");

	// if we couldn't load the resource, print an error messge to the console
	if(!result)
		GePrint("Dialog Example: could not load the resource file.");

	// return status result to C4D
	return result;
}

Bool DialogExample::InitValues(void)
{
	// here we can initialise any variables the dialog will use
	BaseDocument *doc = nullptr;
	Int32 mode;

	// first call the base class function and return FALSE if there was an error
	if(!GeDialog::InitValues())
		return FALSE;

	// we want to set the gadgets to the current values of the object, so first we need to get the active object
	op = nullptr;
	doc = GetActiveDocument();
	if(doc)
		op = doc->GetActiveObject();

	// we can now initialise the dialog gadgets
	// set default values if there's no active object (we shouldn't even be here if there's no active object, but just in case...)
	if(!op)
	{
		SetString(IDC_OBJECTNAME, "");
		SetInt32(IDC_EDITORVIS, 6050);
		SetInt32(IDC_RENDERVIS, 6060);
	}
	else
	{
		// since we do have an object, get the current object values and set the gadgets to those values
		SetString(IDC_OBJECTNAME, op->GetName());
		// get the current editor visiblity
		mode = op->GetEditorMode();
		// we know that 'mode' will have the value 0, 1, or 2 - so we just add that to the starting index for the combo box entries to reflect the current setting
		// the combo box IDs were set in CreateLayout()
		SetInt32(IDC_EDITORVIS, 6050 + mode);
		// get the current render visiblity
		mode = op->GetRenderMode();
		SetInt32(IDC_RENDERVIS, 6060 + mode);
	}
	
	return true;
}

Bool DialogExample::Command(Int32 id, const BaseContainer& msg)
{
	// this is the core of the dialog, which handles user interaction
	// the 'id' parameter is the ID value of the gadget/menu item clicked, changed, etc.
	String sName;
	Int32 mode;
	BaseDocument *doc;

	switch(id)
	{
	case 1:			// this is the OK button, so we make the changes to the object
		if(op)		// make sure we have a valid pointer to the active object
		{
			// get the active document so we can implement undos
			doc = GetActiveDocument();
			doc->StartUndo();						// start adding undos, so that every change we make will be regarded as one undo if the user hits Ctrl/Cmd-Z
			GetString(IDC_OBJECTNAME, sName);		// get the string from the edit field in the dialog box
			// don't set the object name if the edit field was empty, as we don't want an object with no name
			if(sName.Content())
			{
				doc->AddUndo(UNDOTYPE_CHANGE, op);		// add an undo
				op->SetName(sName);
			}
			// get the selected editor mode from the relevant combo box in the dialog
			GetInt32(IDC_EDITORVIS, mode);			// mode will have a value of 6050, 6051, or 6052 as these are the ID values we set in CreateLayout()
			doc->AddUndo(UNDOTYPE_CHANGE, op);
			op->SetEditorMode(mode - 6050);			// SetEditorMode() expects a value from 0 - 2
			// do the same for the render mode
			GetInt32(IDC_RENDERVIS, mode);
			doc->AddUndo(UNDOTYPE_CHANGE, op);
			op->SetRenderMode(mode - 6060);
			doc->EndUndo();							// we can stop adding undos
			op->Message(MSG_UPDATE);
			EventAdd(EVENT_FORCEREDRAW);			// tell C4D to update the display, otherwise it will look as if nothing has changed
		}
		Close();									// close the dialog box
		break;

	case 2:			// this is the cancel button
		Close();	// just close the dialog without making any changes
		break;

	default:		// any other gadget which is changed will just print the ID to the console - this is just for demonstration purposes!
		GePrint("ID of gadget changed: " + String::IntToString(id));
		break;
	}
	return true;
}

Bool DialogExample::AskClose(void)
{
	return FALSE;			// you can set this to true to stop the dialog being closed if you need to in certain circumstances
}

/* **************************************************************
   Definitions of the 'MenuPlug' class derived from 'CommandData'
   
   This is the class which is the 'actual plugin' and which
   opens the dialog where all the user interaction is handled
*****************************************************************/

Bool MenuPlug::Execute(BaseDocument *doc)
{
	// this function is caled whenever the user clicks the name of the plugin in the Plugins menu
	// all we do here is open the dialog modally

	return dlg.Open(DLG_TYPE_MODAL, ID_DIALOGEX, -1, -1, 300, 150, 0);		// the first parameter will open the dialog non-modally if set to true, but we don't want that here
}

Int32 MenuPlug::GetState(BaseDocument *doc)
{
	// return how the plugin name is to be displayed in the Plugins menu:
	// if there is an object currently selected we want to enable the command,
	// but we want it to be disabled if there is no active object

	if(doc->GetActiveObject() != nullptr)
		return CMD_ENABLED;
	else
		return 0;
}

/* ****************************
	Other functions
******************************/

Bool RegisterDialogExample(void)
{
	// decide by name if the plugin shall be registered - just for user convenience
	String sName, sHelp;

	sName = GeLoadString(IDS_DIALOGEX);								// load the plugin name from the global string table - this will appear in the Plugins menu
	sHelp = "Example CommandData plugin with a dialog box";			// this string appears in the status bar when the user hovers over the plugin name in the menu (could be loaded from the string table, of course)

	// register the plugin - note that we supplied the name of the icon to appear in the Plugins menu (C4D will substitute a white question mark if we don't provide one)
	return RegisterCommandPlugin(ID_DIALOGEX, sName, 0, AutoBitmap("dialogex.tif"), sHelp, NewObjClear(MenuPlug));
}
