/************************************************************************/
/* ToggleGI - plugin to toggle global illumination on and off           */
/* This plugin is open source - feel free to modify, change, etc. as	*/
/* much as you like.													*/
/*																		*/
/* By Steve Pedler, February 16th 2011									*/
/************************************************************************/

// ToggleGI - togglegi.cpp

/************************************************************************/
/* Include files                                                        */
/************************************************************************/
#include "c4d.h"
#include "togglegi.h"

/************************************************************************/
/* C4D will call Execute() whenever the user selects the plugin			*/
/* from the Plugins menu                                                */
/************************************************************************/
Bool ToggleGI::Execute(BaseDocument *doc)
{
	// 
	DoToggleGI(doc);		// toggle GI on/off
	return true;
}

/************************************************************************/
/* GetState() tells C4D whether to display the menu item as enabled		*/
/* or disabled; return CMD_ENABLED for the enabled state				*/
/************************************************************************/
Int32 ToggleGI::GetState(BaseDocument *doc)
{
	return CMD_ENABLED;		// the plugin is always available
}

/************************************************************************/
/* This function is the core of the plugin, and will carry out the      */
/* action of toggling GI on and off										*/
/*																		*/
/* Parameters: BaseDocument* doc (the active document)					*/
/* Returns: nothing														*/
/************************************************************************/
void ToggleGI::DoToggleGI(BaseDocument* doc)
{
	RenderData *rdata = nullptr;
	BaseVideoPost *pvp = nullptr;
	Bool foundGI = false;

	rdata = doc->GetActiveRenderData();				// get the render data settings for this document
	if(rdata)										// valid pointer?
	{
		doc->StartUndo();							// add subsequent actions to Cinema's undo list
		// first, we need to check whether GI is already in the active render data settings
		pvp = rdata->GetFirstVideoPost();			// GI is implemented as a videopost effect (VP)
		while(pvp)
		{
			if(pvp->GetType() == VPglobalillumination)			// is this videopost the GI VP?
			{
				foundGI = true;									// yes, so GI is already added to the effects list and we don't need to do that
				break;
			}
			pvp = pvp->GetNext();
		}
		if(!foundGI)			// we didn't find GI, so add it to the render data list of VPs
		{
			pvp = BaseVideoPost::Alloc(VPglobalillumination);
			if(pvp)				// allocated successfully?
			{
				rdata->InsertVideoPost(pvp, nullptr);				// add the VP to the render settings
				doc->AddUndo(UNDOTYPE_NEW, pvp);					// add an undo level
				rdata->Message(MSG_UPDATE);						// tell Cinema we've done something to the render settings
			}
		}
		else					// GI is already there, so all we need to do is to toggle its 'off' bit
		{
			doc->AddUndo(UNDOTYPE_CHANGE, pvp);						// add an undo level
			pvp->ToggleBit(BIT_VPDISABLED);						// this C4D function conveniently toggles the VP on and off
			rdata->Message(MSG_UPDATE);							// tell Cinema we've done something to the render settings
		}
		doc->EndUndo();											// finished adding undos
		EventAdd(EVENT_0);
	}
	else
	{
		// print an error message to the C4D console if for some bizarre reason we couldn't access the render data
		GePrint("ToggleGI: Error - could not obtain the active render data to set up GI.");
	}
}

/************************************************************************/
/* Register the plugin with C4D                                         */
/************************************************************************/
Bool RegisterToggleGI(void)
{
	// decide by name if the plugin shall be registered - just for user convenience
	String sName, sHelp;

	sName = "Toggle Global Illumination";					// this string appears in the Plugins menu
	sHelp = "Turns Global Illumination on or off";			// this string appears in the C4D status bar when the user hovers over the plugin name in the Plugins menu

	// register the plugin
	return RegisterCommandPlugin(ID_TOGGLEGI, sName, 0, AutoBitmap("togglegi.tif"), sHelp, NewObjClear(ToggleGI));
}


