// sIBLLoader - sibl.cpp
// compile for R11

#include <stdio.h>
#include "c4d.h"
#include "c4d_symbols.h"
#include "sibl.h"
#include "tVrayLightTag.h"
#include "tVrayCompositingTag.h"
#include "vpvraybridge.h"
#include "xslafilter.h"
#include "mVrayAdvancedMaterial.h"
#include "bmflip.h"
#include "xbmflip.h"

// ------------------- start of dialog functions ----------------------
// dialog class implementation
SIBLDialog::SIBLDialog(void)
{
	// constructor
	BaseContainer *bc;
	Bool lwfFlag = FALSE;
	BaseDocument *doc = GetActiveDocument();

	// get the document's LWF switch
	bc = doc->GetDataInstance();
	lwfFlag = bc->GetBool(DOCUMENT_LINEARWORKFLOW);

	if(GeGetCurrentOS() == OPERATINGSYSTEM_WIN)
		winFlag = TRUE;
	else
		winFlag = FALSE;
	// initialise the dialog box values and GUI
	ResetOpts();
	// initialise the vray GI settings structure
	vrayGIOpts.lwfHistory = FALSE;
	vrayGIOpts.burn = 1.0;
	vrayGIOpts.cmgamma = 1.0;
	vrayGIOpts.multip = 1.0;
	vrayGIOpts.type = VP_VRAYBRIDGE_COLORMAPPING_TYPE_0;
	vrayGIOpts.lwfOn = VRAY_LWF_NOCHANGE;
	vrayGIOpts.r12LWF = lwfFlag;

	currentColl = -1L;					// no current collection yet
}

SIBLDialog::~SIBLDialog(void)
{
	// delete any memory used for preset data and collections
	FreeMemory();
	dlg_res.Free();
}

void SIBLDialog::FreeMemory(void)
{
	LONG i;
	lightData *lData;
	lsData *lsd;

	for(i = 0; i< presetCount; i++)
	{
		lData = presets[i].lights;
		bDelete(lData);
		lsd = presets[i].ls;
		bDelete(lsd);
	}
	// free up the presets
	presetCount = 0L;
	bDelete(presets);
	presetIndex = -1L;
	oldPresetCount = 0L;
	bDelete(oldPresets);

	// free data used for collections
	bDelete(siblCollecs);
}

Bool SIBLDialog::CreateLayout(void)
{
	Bool res = TRUE;
	BasePlugin *vpbridge = NULL;

	// check for alternative render engines
	// is Vray available?
	vpbridge = FindPlugin(VPvray, PLUGINTYPE_VIDEOPOST);
	if(vpbridge)
		vrayFlag = TRUE;
	else{
		vrayFlag = FALSE;
	}

	// call base class function first
	GeDialog::CreateLayout();

	// create a resource object and initialise it to the plugin's path
	//GeResource dlg_res;
	dlg_res.Init(GeGetPluginPath());
	if(res = LoadDialogResource(IDD_SIBLDIALOG, &dlg_res, 0))
	{
		// set dialog title
		SetTitle("sIBL Loader for Cinema 4D, version 1.5");
		// try to find the bitmap button control
		bbcg = NULL;
		bbcg = (BitmapButtonCustomGui*)FindCustomGui(IDC_THUMB, 1000479);
		if(!bbcg)
			GePrint("sIBL Loader: error in creating interface, thumbnails will not be visible!");
		// find the hyperlink control
		hlcg = NULL;
		hlcg = (HyperLinkCustomGui*)FindCustomGui(IDC_PRESET_LINK, 1009180);
		if(!hlcg)
			GePrint("sIBL Loader: error in creating interface, hyperlinks will not be visible!");
		fnibled = NULL;
		fnibled = (FilenameCustomGui*)FindCustomGui(IDC_SIBLEDITPATH, 1000478);

		// now attach the list view control
		lvPresets.AttachListView(this, IDC_PRESETLIST);

		// attach the user areas
		AttachUserArea(bannerUA, IDC_BANNER, USERAREA_COREMESSAGE);
		AttachUserArea(comfieldUA, IDC_PRESET_COMMENT, USERAREA_COREMESSAGE);
		AttachUserArea(suncolorUA, IDC_SUNCOLOR, USERAREA_COREMESSAGE);

		// add some menu items
		MenuFlushAll();
		MenuSubBegin("File");
		MenuAddString(6012, "Apply preset");
		MenuAddString(6013, "Clear sIBL preset from scene");
		if(winFlag)
			MenuAddString(6014, "Edit with sIBL-Edit");
		MenuAddSeparator();
		MenuAddString(6003, "Add new collection...");
		MenuAddString(6004, "Remove current collection...");
		MenuAddString(6005, "Reload collection");
		MenuAddString(6006, "Reload preset");
		MenuAddSeparator();
		MenuAddString(6007, "Save preferences");
		MenuAddString(6010, "Close Manager");
		MenuSubEnd();
		MenuSubBegin("Help");
		MenuAddString(6020, "Show help file (PDF)");
		MenuAddString(6030, "About...");
		MenuSubEnd();
		MenuFinished();

		// set up resolution combo box
		FreeChildren(IDC_TEXPREVIEW);
		AddChild(IDC_TEXPREVIEW, 7000, "Default");
		AddChild(IDC_TEXPREVIEW, 7001, "64 x 64");
		AddChild(IDC_TEXPREVIEW, 7002, "128 x 128");
		AddChild(IDC_TEXPREVIEW, 7003, "256 x 256");
		AddChild(IDC_TEXPREVIEW, 7004, "512 x 512");
		AddChild(IDC_TEXPREVIEW, 7005, "1024 x 1024");
		AddChild(IDC_TEXPREVIEW, 7006, "2048 x 2048");
		AddChild(IDC_TEXPREVIEW, 7007, "4096 x 4096");
		AddChild(IDC_TEXPREVIEW, 7008, "8192 x 8192");

		// set up the light type combo box
		FreeChildren(IDC_LIGHTTYPE);
		AddChild(IDC_LIGHTTYPE, SPOTLIGHT, "Spots");
		AddChild(IDC_LIGHTTYPE, OMNILIGHT, "Omnis");

		// set up the render engine combo box
		FreeChildren(IDC_RENDERENGINE);
		AddChild(IDC_RENDERENGINE, ADV_RENDER, "Inbuilt renderer");
		if(vrayFlag)
			AddChild(IDC_RENDERENGINE, VRAY_RENDER, "Vray for C4D");

		// set up Vray LWF combo box
		FreeChildren(IDC_VRAY_LWF);
		AddChild(IDC_VRAY_LWF, VRAY_LWF_NOCHANGE, "Do not change color mapping");
		AddChild(IDC_VRAY_LWF, VRAY_LWF_OFF, "Linear workflow disabled");
		AddChild(IDC_VRAY_LWF, VRAY_LWF_8BIT, "8-bit linear workflow");
		AddChild(IDC_VRAY_LWF, VRAY_LWF_32BIT, "32-bit linear workflow");

		// set up the Vray Lightsmith mode box
		FreeChildren(IDC_VRAY_LS_MODE);
		AddChild(IDC_VRAY_LS_MODE, VRAY_LS_MODE_LUM, "BRDF material luminosity");
		AddChild(IDC_VRAY_LS_MODE, VRAY_LS_MODE_DI, "BRDF material direct illumination");
		//AddChild(IDC_VRAY_LS_MODE, VRAY_LS_MODE_MESH, "Simple mesh light");

		// initialise dialog gadgets here
		SetLong(IDC_RADIOGROUPFILTER, IDC_RADIOFILTERANY);
		SetLong(IDC_RADIOGROUPSEARCH, IDC_RADIOSEARCHCOMMENT);
		SetOpts();
		SetupGUI();
	}
	else
	{
		// print error to console if there's a resource problem
		GePrint("sIBL Loader: could not load resource file.");
	}

	return res;
}

void SIBLDialog::ResetOpts(void)
{
	theOpts.loadBG = TRUE;
	theOpts.loadENV = TRUE;
	theOpts.loadREF = TRUE;
	theOpts.loadSUN = TRUE;
	theOpts.correctSun = TRUE;
	theOpts.loadLights = TRUE;
	theOpts.lightType = SPOTLIGHT;			// spots by default
	theOpts.renderEngine = ADV_RENDER;		// use AR by default
	theOpts.alignNorth = TRUE;
	//theOpts.doLWF = FALSE;
	theOpts.setupGI = TRUE;
	theOpts.overWriteExisting = FALSE;
	theOpts.turnOffLights = TRUE;
	theOpts.useSpheres = FALSE;
	theOpts.sphereRadius = 5000.0;
	theOpts.texPreview = 7000;
	theOpts.autoSavePrefs = FALSE;
	theOpts.maxCollecs = DEFAULT_MAX_COLLECTIONS;
	theOpts.loadAll = FALSE;
	theOpts.filterOnName = FALSE;
	theOpts.filterAll = FALSE;
	theOpts.warnNoMatches = FALSE;
	theOpts.vrayUseGIEnv = FALSE;
	theOpts.vrayUseSunIntensity = TRUE;
	theOpts.vrayEditorBGSky = FALSE;
	theOpts.vrayWarnBG = TRUE;
	theOpts.vrayAddRefraction = TRUE;
	theOpts.vrayLWFMode = VRAY_LWF_NOCHANGE;
	theOpts.warnOnClear = TRUE;
	theOpts.saveDefCollect = FALSE;
	theOpts.ls_scalemulti = 50.0;
	theOpts.vrayLSMode = VRAY_LS_MODE_DI;
	theOpts.vrayLSIntensMulti = 2.0;
}

void SIBLDialog::GetOpts(void)
{
	LONG radFilter;

	// gets the values from the dialog box into the options structure
	GetBool(IDC_LOADBG, theOpts.loadBG);
	GetBool(IDC_LOADENVIRON, theOpts.loadENV);
	GetBool(IDC_LOADREFLEC, theOpts.loadREF);
	GetBool(IDC_LOADSUN, theOpts.loadSUN);
	GetBool(IDC_CORRECTSUN, theOpts.correctSun);
	GetBool(IDC_LOADLIGHTS, theOpts.loadLights);
	GetLong(IDC_LIGHTTYPE, theOpts.lightType);
	GetLong(IDC_RENDERENGINE, theOpts.renderEngine);
	GetBool(IDC_ALIGNNORTH, theOpts.alignNorth);
	//GetBool(IDC_DOLWF, theOpts.doLWF);
	GetBool(IDC_SETUPGI, theOpts.setupGI);
	GetBool(IDC_OVERWRITE, theOpts.overWriteExisting);
	GetBool(IDC_KILLLIGHTS, theOpts.turnOffLights);
	GetLong(IDC_MAXCOLLECTIONS, theOpts.maxCollecs);
	GetBool(IDC_USESPHERES, theOpts.useSpheres);
	GetReal(IDC_DOMERADIUS, theOpts.sphereRadius);
	GetLong(IDC_TEXPREVIEW, theOpts.texPreview);
	GetBool(IDC_AUTOSAVEPREFS, theOpts.autoSavePrefs);
	GetBool(IDC_LOADALL, theOpts.loadAll);
	GetLong(IDC_RADIOGROUPSEARCH, radFilter);
	if(radFilter == IDC_RADIOSEARCHCOMMENT)
		theOpts.filterOnName = FALSE;
	else
		theOpts.filterOnName = TRUE;
	GetLong(IDC_RADIOGROUPFILTER, radFilter);
	if(radFilter == IDC_RADIOFILTERALL)
		theOpts.filterAll = TRUE;
	else
		theOpts.filterAll = FALSE;
	GetBool(IDC_WARNNOMATCHES, theOpts.warnNoMatches);
	GetBool(IDC_VRAY_USEGIENV, theOpts.vrayUseGIEnv);
	GetBool(IDC_VRAY_SUNINTENSITY, theOpts.vrayUseSunIntensity);
	GetBool(IDC_VRAY_EDITORBGSKY, theOpts.vrayEditorBGSky);
	GetBool(IDC_VRAYBGWARNING, theOpts.vrayWarnBG);
	GetBool(IDC_VRAY_ADDREFRAC, theOpts.vrayAddRefraction);
	GetLong(IDC_VRAY_LWF, theOpts.vrayLWFMode);
	GetBool(IDC_WARNONCLEAR, theOpts.warnOnClear);
	GetBool(IDC_SETDEFAULTCOLL, theOpts.saveDefCollect);
	GetReal(IDC_LS_SCALE, theOpts.ls_scalemulti);
	GetLong(IDC_VRAY_LS_MODE, theOpts.vrayLSMode);
	GetReal(IDC_VRAY_LSMULT, theOpts.vrayLSIntensMulti);
}

void SIBLDialog::SetOpts(void)
{
	// loads the values from the options structure (initialised in the constructor) into the dialog box
	SetBool(IDC_LOADBG, theOpts.loadBG);
	SetBool(IDC_LOADENVIRON, theOpts.loadENV);
	SetBool(IDC_LOADREFLEC, theOpts.loadREF);
	SetBool(IDC_LOADSUN, theOpts.loadSUN);
	SetBool(IDC_CORRECTSUN, theOpts.correctSun);
	SetBool(IDC_LOADLIGHTS, theOpts.loadLights);
	SetLong(IDC_LIGHTTYPE, theOpts.lightType);
	SetLong(IDC_RENDERENGINE, theOpts.renderEngine);
	SetBool(IDC_ALIGNNORTH, theOpts.alignNorth);
	//SetBool(IDC_DOLWF, theOpts.doLWF);
	SetBool(IDC_SETUPGI, theOpts.setupGI);
	SetBool(IDC_OVERWRITE, theOpts.overWriteExisting);
	SetBool(IDC_KILLLIGHTS, theOpts.turnOffLights);
	SetLong(IDC_MAXCOLLECTIONS, theOpts.maxCollecs, 1, MAXLONGl);
	SetBool(IDC_USESPHERES, theOpts.useSpheres);
	SetMeter(IDC_DOMERADIUS, theOpts.sphereRadius, 0, MAXREALr);
	SetLong(IDC_TEXPREVIEW, theOpts.texPreview);
	SetBool(IDC_AUTOSAVEPREFS, theOpts.autoSavePrefs);
	SetBool(IDC_LOADALL, theOpts.loadAll);
	if(theOpts.filterOnName)
		SetLong(IDC_RADIOGROUPSEARCH, IDC_RADIOSEARCHNAME);
	else
		SetLong(IDC_RADIOGROUPSEARCH, IDC_RADIOSEARCHCOMMENT);
	if(theOpts.filterAll)
		SetLong(IDC_RADIOGROUPFILTER, IDC_RADIOFILTERALL);
	else
		SetLong(IDC_RADIOGROUPFILTER, IDC_RADIOFILTERANY);
	SetBool(IDC_WARNNOMATCHES, theOpts.warnNoMatches);
	SetBool(IDC_VRAY_USEGIENV, theOpts.vrayUseGIEnv);
	SetBool(IDC_VRAY_SUNINTENSITY, theOpts.vrayUseSunIntensity);
	SetBool(IDC_VRAY_EDITORBGSKY, theOpts.vrayEditorBGSky);
	SetBool(IDC_VRAYBGWARNING, theOpts.vrayWarnBG);
	SetBool(IDC_VRAY_ADDREFRAC, theOpts.vrayAddRefraction);
	SetLong(IDC_VRAY_LWF, theOpts.vrayLWFMode);
	SetBool(IDC_WARNONCLEAR, theOpts.warnOnClear);
	SetBool(IDC_SETDEFAULTCOLL, theOpts.saveDefCollect);
	SetReal(IDC_LS_SCALE, theOpts.ls_scalemulti, 0, MAXREALr, 0.1, FORMAT_REAL);
	SetLong(IDC_VRAY_LS_MODE, theOpts.vrayLSMode);
	SetReal(IDC_VRAY_LSMULT, theOpts.vrayLSIntensMulti, 0.1, MAXREALr, 0.1, FORMAT_REAL);
}

void SIBLDialog::SetupGUI(void)
{
	LONG r, g, b;
	String labelText;
	BaseBitmap *bm;
	BaseContainer msg;

	// is OS Windows?
	if(winFlag)
	{
		Enable(IDC_SIBLEDIT, TRUE);
		Enable(IDC_SIBLEDITPATH, TRUE);
	}
	else
	{
		Enable(IDC_SIBLEDIT, FALSE);
		Enable(IDC_SIBLEDITPATH, FALSE);
	}

	// is Vray available?
	if(!vrayFlag)
	{
		Enable(IDC_VRAY_ADDREFRAC, FALSE);
		Enable(IDC_VRAY_SUNINTENSITY, FALSE);
		Enable(IDC_VRAY_USEGIENV, FALSE);
		Enable(IDC_VRAY_GROUNDOBJECT, FALSE);
		Enable(IDC_VRAY_EDITORBGSKY, FALSE);
	}


	if(theOpts.renderEngine == VRAY_RENDER && vrayFlag)		// use Vray
	{
		UpdateLayout(theOpts.renderEngine);
	}
	else									// must be AR
	{
		UpdateLayout(theOpts.renderEngine);
	}

	if(theOpts.loadAll)
	{
		Enable(IDC_COLLECTIONS, FALSE);
		Enable(IDC_CURRENTCOLL, FALSE);
	}
	else
	{
		Enable(IDC_COLLECTIONS, TRUE);
		Enable(IDC_CURRENTCOLL, TRUE);
	}

	if(filterFlag)						// if presets are being filtered, turn off 'Apply filter' and enable 'Remove filter'
	{
		Enable(IDC_REMOVEFILTER, TRUE);
		Enable(IDC_DOFILTER, FALSE);
		Enable(IDC_KEYWORDS, FALSE);
		Enable(IDC_RADIOGROUPFILTER, FALSE);
		Enable(IDC_RADIOGROUPSEARCH, FALSE);
	}
	else
	{
		Enable(IDC_REMOVEFILTER, FALSE);
		Enable(IDC_DOFILTER, TRUE);
		Enable(IDC_KEYWORDS, TRUE);
		Enable(IDC_RADIOGROUPFILTER, TRUE);
		Enable(IDC_RADIOGROUPSEARCH, TRUE);
	}

	// clear preset comment
	comfieldUA.txtComment = "";
	// set colorfield to background
	GetColorRGB(COLOR_BG, r, b, g);
	suncolorUA.color = Vector((Real)r/255, (Real)b/255, (Real)g/255);
	// disable the sphere radius edit box as the default state is not to use spheres
	Enable(IDC_DOMERADIUS, FALSE);
	// remove the thumb bitmap?
	bm = NULL;
	if(bbcg)
		bbcg->SetImage(bm, FALSE);
	// clear link
	labelText = "";
	if(hlcg)
		hlcg->SetLinkString(&labelText, &labelText);
	// set name
	SetString(IDC_PRESET_NAME, "");
	// set author
	SetString(IDC_PRESET_AUTHOR, "");
	// set location
	SetString(IDC_PRESET_LOCATION, "");	
	// set latitude and longtitude
	SetString(IDC_LATITUDE, "");
	SetString(IDC_LONGTITUDE, "");
	// set date and time
	SetString(IDC_DATE, "");
	SetString(IDC_TIME, "");
	msg.SetId(BFM_SETSTATUSBAR);
	msg.SetString(BFM_STATUSBAR_TXT, "");
	SendMessage(IDC_SCROLLTHUMBS, msg);

	return;
}

void SIBLDialog::UpdateLayout(LONG renderEngine)
{
	// updates the interface depending on the render engine selected
	switch(renderEngine)
	{
	case VRAY_RENDER:
		// enable Vray options
		Enable(IDC_VRAY_ADDREFRAC, TRUE);
		Enable(IDC_VRAY_SUNINTENSITY, TRUE);
		Enable(IDC_VRAY_USEGIENV, TRUE);
		Enable(IDC_VRAY_GROUNDOBJECT, TRUE);
		Enable(IDC_VRAY_EDITORBGSKY, TRUE);
		Enable(IDC_VRAY_LWF, TRUE);
		Enable(IDC_VRAY_LS_MODE, TRUE);
		Enable(IDC_VRAY_LSMULT, TRUE);
		if(!theOpts.loadREF)
			Enable(IDC_VRAY_ADDREFRAC, FALSE);
		else
			Enable(IDC_VRAY_ADDREFRAC, TRUE);
		// disable use spheres
		Enable(IDC_USESPHERES, FALSE);
		Enable(IDC_DOMERADIUS, FALSE);
		// enable GI and LWF options
		Enable(IDC_SETUPGI, TRUE);
		// enable light and other options
		Enable(IDC_LOADSUN, TRUE);
		Enable(IDC_LOADLIGHTS, TRUE);
		Enable(IDC_LIGHTTYPE, TRUE);
		Enable(IDC_KILLLIGHTS, TRUE);
		Enable(IDC_OVERWRITE, TRUE);
		SetBool(IDC_LOADSUN, FALSE);
		SetBool(IDC_VRAY_SUNINTENSITY, FALSE);
		Enable(IDC_VRAY_SUNINTENSITY, FALSE);
		break;

	case ADV_RENDER:
		// disable Vray options
		Enable(IDC_VRAY_ADDREFRAC, FALSE);
		Enable(IDC_VRAY_SUNINTENSITY, FALSE);
		Enable(IDC_VRAY_USEGIENV, FALSE);
		Enable(IDC_VRAY_GROUNDOBJECT, FALSE);
		Enable(IDC_VRAY_EDITORBGSKY, FALSE);
		Enable(IDC_VRAY_LWF, FALSE);
		Enable(IDC_VRAY_LS_MODE, FALSE);
		Enable(IDC_VRAY_LSMULT, FALSE);
		// enable use spheres, and the dome radius if the user has selected that
		Enable(IDC_USESPHERES, TRUE);
		if(theOpts.useSpheres)
			Enable(IDC_DOMERADIUS, TRUE);
		// enable GI and LWF options
		Enable(IDC_SETUPGI, TRUE);
		// enable light and other options
		Enable(IDC_LOADSUN, TRUE);
		SetBool(IDC_LOADSUN, TRUE);
		Enable(IDC_LOADLIGHTS, TRUE);
		Enable(IDC_LIGHTTYPE, TRUE);
		Enable(IDC_KILLLIGHTS, TRUE);
		Enable(IDC_OVERWRITE, TRUE);
		break;
	}
}

Bool SIBLDialog::InitValues(void)
{
	BaseContainer lvLayout;
	
	// first call the base class function
	if(!GeDialog::InitValues()){
		return FALSE;
	}
	
	// allocate memory for collections
	numColls = 0;						// no collections in list
	theOpts.maxCollecs = DEFAULT_MAX_COLLECTIONS;			// maximum of 25 collections by default

	sibleditFolder = Filename();		// empty Filename for sIBL-Edit location
	presetCount = 0L;					// no presets loaded yet
	presetIndex = -1L;
	presets = NULL;
	oldPresetCount = 0L;
	oldPresets = NULL;
	filterFlag = FALSE;					// filter being applied yet
	lvLayout.SetLong('name', LV_COLUMN_TEXT);
	lvPresets.SetLayout(1, lvLayout);
	LoadPrefs();						// load global prefs

	if(numColls > 0)				// if there is at least one collection, scan it or all of them
	{
		ScanCollection(TRUE);
		if(presetCount > 0)
		{
			presetIndex = 0L;
			ShowSetDetails(presetIndex);
		}
	}

	return TRUE;
}

Bool SIBLDialog::AskClose(void)
{
	LONG i;

	GetOpts();

	//clear the list view of any existing content
	for (i = lvPresets.GetItemCount()-1; i>=0; i--)
	{
		lvPresets.RemoveItem(i);
	}

	// automatically save the preferences if that option is selected, or if not just save the collection paths
	if(theOpts.autoSavePrefs)
		SavePrefs();
	else
		SaveCollectionsPrefs();

	// free data used for presets and collections
	FreeMemory();

	return FALSE;
}

Bool SIBLDialog::Command(LONG id, const BaseContainer &msg)
{
	// show command value
	Filename filHelp;
	LONG radFilter;
	LONG old_engine;
	Bool answer;
	BaseDocument *doc;

	switch (id)
	{
	case IDC_APPLYPRESET:		// Apply preset button
	case 6012:					// menu entry
		// call function to set up the IBL
		GetOpts();		// get the user options in case we missed any 
		if(presetCount > 0)
		{
			if(theOpts.renderEngine == VRAY_RENDER && vrayFlag)
				ApplyPresetVray();
			else
				ApplyPreset();
			Close();			// close dialog after applying preset
		}
		break;

	case IDC_CLOSE:		// Close button
	case 6010:			// File menu Close Manager
		Close();
		break;
			
	case 6005:			// rescan the current collection, or all of them
	case IDC_SCANCOLLECTION:
		if(numColls > 0)
		{
			ScanCollection(TRUE);
			if(presetCount > 0)
			{
				presetIndex = 0L;
				ShowSetDetails(presetIndex);
			}
		}
		break;

	case 6004:			// remove the current collection
	case IDC_REMOVECOLLECTION:
		if(QuestionDialog("Are you sure you want to remove this collection of sIBL sets from the list of collections?"))
			RemoveCollection();
		break;

	case 6003:
	case IDC_LOADCOLLECTION:						// add a new collection
		if(numColls < theOpts.maxCollecs)			// must be enough room to add a collection
		{
			adcl.Open();
			if(adcl.buttonClicked)					// clicked OK on the dialog
			{
				siblCollecs[numColls].fname = adcl.newCollectionFolder;
				siblCollecs[numColls].name = adcl.newCollectionName;
				AddChild(IDC_COLLECTIONS, numColls, siblCollecs[numColls].name);
				AddChild(IDC_CURRENTCOLL, numColls, siblCollecs[numColls].name);
				numColls++;						// increment collections count
				currentColl = numColls - 1;		// set to the new collection
				SetLong(IDC_COLLECTIONS, currentColl);
				SetLong(IDC_CURRENTCOLL, currentColl);
				if(filterFlag)
				{
					RemoveFilter(FALSE);
					ScanCollection(FALSE);
					ApplyFilter(TRUE);
				}
				else
					ScanCollection(TRUE);
				if(presetCount > 0)
				{
					presetIndex = 0L;
					ShowSetDetails(presetIndex);
					SetLong(IDC_TABS, IDC_TAB1);		// change to browser tab
				}
			}
		}
		else
		{
			MessageDialog("You have " + LongToString(numColls) + " collections already. Please remove one or more collections, or increase the maximum number of collections. In each case you will need to save your preferences and restart the plugin before adding a collection.");
		}
		break;

	case 6007:		// save preferences
	case IDC_SAVEPREFS:
		SavePrefs();
		break;

	case 6020:	// show help file
		filHelp = GeGetPluginPath() + Filename("sIBLLoader.pdf");
		GeExecuteFile(filHelp.GetString());		
		break;

	case 6030:	// About box
		vrs.Open();
		break;
			
	case IDC_PRESETLIST:
		switch (msg.GetLong(BFM_ACTION_VALUE))
		{
			case LV_SIMPLE_FOCUSITEM:
				presetIndex = msg.GetLong(LV_SIMPLE_ITEM_ID);
				ShowSetDetails(presetIndex);
				break;
		}
		break;

	case IDC_SHOWPREVIEW:
		prv.SetImageFile(presets[presetIndex].path + Filename(presets[presetIndex].previewFile));
		prv.SetImagename(presets[presetIndex].name);
		prv.Open();
		break;

	// options panel
	case IDC_LOADBG:
		GetBool(IDC_LOADBG, theOpts.loadBG);
		if(vrayFlag)
		{
			if(theOpts.loadBG)			// if we have Vray, turn off the editor BG option if the user doesn't want a background
				Enable(IDC_VRAY_EDITORBGSKY, TRUE);
			else
				Enable(IDC_VRAY_EDITORBGSKY, FALSE);
		}
		break;

	case IDC_LOADENVIRON:
		GetBool(IDC_LOADENVIRON, theOpts.loadENV);
		// disable Vray option to load the GI environment if Vray is loaded
		if(vrayFlag)
		{
			if(!theOpts.loadENV)
				Enable(IDC_VRAY_USEGIENV, FALSE);
			else
				Enable(IDC_VRAY_USEGIENV, TRUE);
		}

		break;

	case IDC_LOADREFLEC:
		GetBool(IDC_LOADREFLEC, theOpts.loadREF);
		if(theOpts.renderEngine == VRAY_RENDER && !theOpts.loadREF)
			Enable(IDC_VRAY_ADDREFRAC, FALSE);
		if(theOpts.renderEngine == VRAY_RENDER && theOpts.loadREF)
			Enable(IDC_VRAY_ADDREFRAC, TRUE);
		break;

	case IDC_LOADSUN:
		GetBool(IDC_LOADSUN, theOpts.loadSUN);
		if(theOpts.loadSUN && theOpts.renderEngine == VRAY_RENDER && vrayFlag)
			Enable(IDC_VRAY_SUNINTENSITY, TRUE);
		else if(!theOpts.loadSUN)
			Enable(IDC_VRAY_SUNINTENSITY, FALSE);
		break;

	case IDC_LOADLIGHTS:
		GetBool(IDC_LOADLIGHTS, theOpts.loadLights);
		if(theOpts.loadLights)
			Enable(IDC_LIGHTTYPE, TRUE);
		else
			Enable(IDC_LIGHTTYPE, FALSE);
		break;

	case IDC_LIGHTTYPE:
		GetLong(IDC_LIGHTTYPE, theOpts.lightType);
		break;

	case IDC_SETUPGI:
		GetBool(IDC_SETUPGI, theOpts.setupGI);
		break;

	case IDC_KILLLIGHTS:
		GetBool(IDC_KILLLIGHTS, theOpts.turnOffLights);
		break;

	case IDC_ALIGNNORTH:
		GetBool(IDC_ALIGNNORTH, theOpts.alignNorth);
		break;

	case IDC_OVERWRITE:
		GetBool(IDC_OVERWRITE, theOpts.overWriteExisting);
		break;

	case IDC_VRAY_ADDREFRAC:
		GetBool(IDC_VRAY_ADDREFRAC, theOpts.vrayAddRefraction);
		break;

	case IDC_VRAY_SUNINTENSITY:
		GetBool(IDC_VRAY_SUNINTENSITY, theOpts.vrayUseSunIntensity);
		break;

	case IDC_VRAY_USEGIENV:
		GetBool(IDC_VRAY_USEGIENV, theOpts.vrayUseGIEnv);
		break;

		// options panel buttons
	case IDC_SIBLEDIT:		// run sIBL-Edit
	case 6014:				// menu entry
		if(sibleditFolder.GetString() != "")
		{
			if(presetCount > 0)					// if we have presets, edit the current one
			{
				GeExecuteProgram(sibleditFolder + Filename("sibledit.exe"), presets[presetIndex].path + presets[presetIndex].file);
			}
			else								// if there are none, just run sibledit.exe
				GeExecuteProgram(sibleditFolder + Filename("sibledit.exe"), Filename());
		}
		break;

	case IDC_SIBLRELOAD:			// reload current preset file	
	case 6006:						// menu entry
		if(!ReloadFile())
			MessageDialog("sIBL Loader failed to reload the preset file. Has it been moved to another location?");
		break;

	case IDC_REMOVEPRESET:			// remove any sIBL preset(s) from the scene
	case 6013:
		doc = GetActiveDocument();
		answer = TRUE;
		if(theOpts.warnOnClear)
			answer = QuestionDialog("This will clear any sIBL preset(s) from the scene for the currently selected render engine. This option cannot be undone. \n\nAre you sure you want to proceed?");
		if(answer)
		{
			if(doc)
				RemovesIBL(doc, theOpts.renderEngine);
			EventAdd(EVENT_0);
			Close();
		}
		break;

	// filter tab
	case IDC_DOFILTER:
		if(!filterFlag)
			ApplyFilter(TRUE);
		break;

	case IDC_REMOVEFILTER:
		if(filterFlag)
			RemoveFilter(TRUE);
		break;

	case IDC_RADIOGROUPSEARCH:
		GetLong(IDC_RADIOGROUPSEARCH, radFilter);
		if(radFilter == IDC_RADIOSEARCHNAME)
		{
			theOpts.filterOnName = TRUE;
		}
		else
		{
			theOpts.filterOnName = FALSE;
		}
		break;

	case IDC_RADIOGROUPFILTER:
		GetLong(IDC_RADIOGROUPFILTER, radFilter);
		if(radFilter == IDC_RADIOFILTERALL)
		{
			theOpts.filterAll = TRUE;
		}
		else
		{
			theOpts.filterAll = FALSE;
		}
		break;

	case IDC_WARNNOMATCHES:
		GetBool(IDC_WARNNOMATCHES, theOpts.warnNoMatches);
		break;

	// preferences tab
	case IDC_CURRENTCOLL:
		GetLong(IDC_CURRENTCOLL, currentColl);				// get the selected collection
		SetLong(IDC_COLLECTIONS, currentColl);				// synchronise with the other combo
		ChooseCollection();									// apply collection
		break;

	case IDC_SETDEFAULTCOLL:
		GetBool(IDC_SETDEFAULTCOLL, theOpts.saveDefCollect);	// checkbox for saving default collection
		break;

	case IDC_COLLECTIONS:
		GetLong(IDC_COLLECTIONS, currentColl);				// get the selected collection
		SetLong(IDC_CURRENTCOLL, currentColl);				// synchronise with the other combo
		ChooseCollection();									// apply collection
		break;

	case IDC_LOADALL:
		GetBool(IDC_LOADALL, theOpts.loadAll);
		if(theOpts.loadAll)
		{
			Enable(IDC_COLLECTIONS, FALSE);
			Enable(IDC_CURRENTCOLL, FALSE);
		}
		else
		{
			Enable(IDC_COLLECTIONS, TRUE);
			Enable(IDC_CURRENTCOLL, TRUE);
		}
		if(numColls > 0 && (currentColl >= 0 && currentColl < numColls))
		{
			if(filterFlag)
			{
				RemoveFilter(FALSE);
				ScanCollection(FALSE);
				ApplyFilter(TRUE);
			}
			else
				ScanCollection(TRUE);
			if(presetCount > 0)
			{
				presetIndex = 0L;
				ShowSetDetails(presetIndex);
			}
		}		
		break;

	case IDC_SIBLEDITPATH:
		if(fnibled)
			sibleditFolder = fnibled->GetData().GetValue().GetFilename();
		break;

	case IDC_RENDERENGINE:
		old_engine = theOpts.renderEngine;
		if(old_engine == VRAY_RENDER && !vrayFlag)
			old_engine = ADV_RENDER;
		GetLong(IDC_RENDERENGINE, theOpts.renderEngine);
		// check that the user hasn't chosen an unavailable render engine
		if(theOpts.renderEngine == VRAY_RENDER && !vrayFlag)
		{
			theOpts.renderEngine = old_engine;
			SetLong(IDC_RENDERENGINE, old_engine);
		}
		if(theOpts.renderEngine == VRAY_RENDER && vrayFlag)
		{
			UpdateLayout(theOpts.renderEngine);
		}
		else
		{
			UpdateLayout(theOpts.renderEngine);
		}
		break;

	case IDC_CORRECTSUN:
		GetBool(IDC_CORRECTSUN, theOpts.correctSun);
		break;

	case IDC_USESPHERES:
		GetBool(IDC_USESPHERES, theOpts.useSpheres);
		Enable(IDC_DOMERADIUS, theOpts.useSpheres);
		break;

	case IDC_DOMERADIUS:
		GetReal(IDC_DOMERADIUS, theOpts.sphereRadius);
		break;

	case IDC_WARNONCLEAR:
		GetBool(IDC_WARNONCLEAR, theOpts.warnOnClear);
		break;

	case IDC_VRAYBGWARNING:
		GetBool(IDC_VRAYBGWARNING, theOpts.vrayWarnBG);
		break;

	default:
		// check for the runtime-added bitmap buttons in the browser
		if(presetCount > 0 && (id >= SIBL_GUI_BITBUTTON && id < SIBL_GUI_BITBUTTON + presetCount))
		{
			presetIndex = id - SIBL_GUI_BITBUTTON;
			ShowSetDetails(presetIndex);
			SetLong(IDC_TABS, IDC_TAB2);		// change to details tab
			AutoAlloc<BaseSelect> sel;
			if(sel)
			{

				sel->DeselectAll();
				sel->Select(id - SIBL_GUI_BITBUTTON);
				lvPresets.SetSelection(sel);
				lvPresets.DataChanged();
			}
		}
		break;
	}
		
	return TRUE;
}

void SIBLDialog::ChooseCollection(void)
{
	if(currentColl >= 0 && currentColl < numColls)		// range check
	{
		if(filterFlag)
		{
			RemoveFilter(FALSE);
			ScanCollection(FALSE);
			ApplyFilter(TRUE);
		}
		else
			ScanCollection(TRUE);
		if(presetCount > 0)
		{
			presetIndex = 0L;
			ShowSetDetails(presetIndex);
		}
	}		
}

void SIBLDialog::ApplyFilter(Bool redrawFlag)
{
	// sort the presets held in memory according to the user's filter
	String filterString, temp;
	presetData *newPresets = NULL;
	tokenData *tokens = NULL;
	LONG numTokens, numMatches, pos, start, i, j, k;
	IniRead ir;
	Bool bMatch;

	// get number of tokens
	GetString(IDC_KEYWORDS, filterString);
	if(!filterString.Content())
	{
		MessageDialog("Please enter one or more comma-separated keywords to search by.");
		return;
	}
	else
	{
		numTokens = 0L;
		start = 0L;
		while(start < filterString.GetLength())
		{
			if(filterString.FindFirst(",", &pos, start))
			{
				// we found the delimiter
				numTokens ++;
				start = pos + 1;
			}
			else
			{
				// no delimiter, so there can only be one string
				numTokens++;
				start = filterString.GetLength();
			}
		}
	}

	// allocate memory for token structures
	if(numTokens > 0)
	{
		tokens = bNew tokenData[numTokens];
		if(tokens)
		{
			// get the tokens
			i = 0;
			start = 0;
			while(start < filterString.GetLength() && i < numTokens)
			{
				if(filterString.FindFirst(",", &pos, start))
				{
					// we found the delimiter
					temp = ir.allTrim(filterString.SubStr(start, pos - start));
					tokens[i].tokenStr = temp.ToLower();
					start = pos + 1;
					i++;
				}
				else
				{
					// no delimiter, so there can only be one string or one substring left in the filter
					temp = ir.allTrim(filterString.SubStr(start, filterString.GetLength() - start));
					tokens[i].tokenStr = temp.ToLower();
					i++;
					start = filterString.GetLength();
				}
			}
			// now count the number of matching presets
			numMatches = 0L;
			if(!theOpts.filterAll)				// include if any keywords
			{
				for(i = 0; i < presetCount; i++)
				{
					if(theOpts.filterOnName)
						temp = presets[i].name.ToLower();
					else
						temp = presets[i].comment.ToLower();
					for (j = 0; j < numTokens; j++)
					{
						if(temp.FindFirst(tokens[j].tokenStr, &pos))
						{
							numMatches++;
							break;					// don't need to iterate through any remaining tokens if we find a match
						}
					}
				}
			}
			else
			{
				for(i = 0; i < presetCount; i++)
				{
					bMatch = TRUE;					// assume it's a match until we don't find a keyword
					if(theOpts.filterOnName)
						temp = presets[i].name.ToLower();
					else
						temp = presets[i].comment.ToLower();
					for (j = 0; j < numTokens; j++)
					{
						if(!temp.FindFirst(tokens[j].tokenStr, &pos))
						{
							bMatch = FALSE;
							break;					// don't need to iterate through any remaining tokens if we find a keyword that doesn't match
						}
					}
					if(bMatch)
						numMatches++;
				}
			}

			// allocate memory for the subset of presets
			newPresets = bNew presetData[numMatches];
			if(newPresets)
			{
				if(!theOpts.filterAll)				// filter on any keyword
				{
					k = 0;
					for(i = 0; i < presetCount; i++)
					{
						if(theOpts.filterOnName)
							temp = presets[i].name.ToLower();
						else
							temp = presets[i].comment.ToLower();
						for (j = 0; j < numTokens; j++)
						{
							if(temp.FindFirst(tokens[j].tokenStr, &pos))
							{
								newPresets[k] = presets[i];		// copy preset over to new array
								k++;
								break;							// don't need to iterate through any remaining tokens if we find a match
							}
						}
					}
				}
				else
				{
					k = 0;
					for(i = 0; i < presetCount; i++)
					{
						bMatch = TRUE;							// assume we have a match until we hit a keyword which doesn't match
						if(theOpts.filterOnName)
							temp = presets[i].name.ToLower();
						else
							temp = presets[i].comment.ToLower();
						for (j = 0; j < numTokens; j++)
						{
							if(!temp.FindFirst(tokens[j].tokenStr, &pos))
							{
								bMatch = FALSE;
								break;							// don't need to iterate through any remaining tokens if we find a match
							}
						}
						if(bMatch)
						{
							newPresets[k] = presets[i];		// copy preset over to new array
							k++;
						}
					}
				}
				// swap over to the new subset and store the unfiltered presets
				filterFlag = TRUE;							// presets are now being filtered
				Enable(IDC_REMOVEFILTER, TRUE);				// so disable applying a filter
				Enable(IDC_DOFILTER, FALSE);
				Enable(IDC_KEYWORDS, FALSE);
				Enable(IDC_RADIOGROUPFILTER, FALSE);
				Enable(IDC_RADIOGROUPSEARCH, FALSE);
				oldPresetCount = presetCount;				// count of unfiltered presets
				oldPresets = presets;						// memory holding unfiltered presets
				presetCount = numMatches;
				presets = newPresets;
				presetIndex = 0L;							// set to first preset in the list
				// clear the list view of any existing content
				for (i = lvPresets.GetItemCount()-1; i>=0; i--)
				{
					lvPresets.RemoveItem(i);
				}
				// clear the thumbnails and set to the new subset
				LayoutFlushGroup(IDC_THUMBNAILS);
				// initialise the GUI
				SetupGUI();
				if(redrawFlag)
					ShowThumbs(numMatches, " preset(s) loaded (*** note: filter is ON ***)");
				SetLong(IDC_TABS, IDC_TAB1);				// switch to browser tab
				if(presetCount > 0)
				{
					presetIndex = 0L;
					ShowSetDetails(presetIndex);
				}
			}
			else
			{
				GePrint("ApplyFilter: could not allocate memory for matched presets.");
			}
			if(numMatches == 0 && theOpts.warnNoMatches)
			{
				if(theOpts.loadAll)
					MessageDialog("No presets matching the filter string were found in the sIBL collections.");
				else
					MessageDialog("No presets matching the filter string were found in the collection " + siblCollecs[currentColl].name + ".");
			}
			// free tokens memory
			bDelete(tokens);
		}
		else
			GePrint("ApplyFilter: could not allocate memory for tokens.");
	}
}

void SIBLDialog::RemoveFilter(Bool redrawFlag)
{
	LONG i;
	
	filterFlag = FALSE;							// filter is no longer being applied
	Enable(IDC_REMOVEFILTER, FALSE);
	Enable(IDC_DOFILTER, TRUE);
	Enable(IDC_KEYWORDS, TRUE);
	presetCount = oldPresetCount;				// reset count to unfiltered presets
	oldPresetCount = 0L;						// count of unfiltered presets
	presets = oldPresets;						// point to unfiltered presets
	oldPresets = NULL;							// memory holding unfiltered presets
	presetIndex = 0L;							// set to first preset in the list
	// clear the list view of any existing content
	for (i = lvPresets.GetItemCount()-1; i>=0; i--)
	{
		lvPresets.RemoveItem(i);
	}
	// clear the thumbnails and set to the new subset
	LayoutFlushGroup(IDC_THUMBNAILS);
	// initialise the GUI
	SetupGUI();
	if(redrawFlag)
		ShowThumbs(presetCount, " preset(s) loaded");
	SetLong(IDC_TABS, IDC_TAB1);				// switch to browser tab
	if(presetCount > 0)
	{
		presetIndex = 0L;
		ShowSetDetails(presetIndex);
	}
}


void SIBLDialog::RemoveCollection(void)
{
	// removes a collection from the list
	LONG collToRemove, i;

	GetLong(IDC_COLLECTIONS, collToRemove);
	if(collToRemove >= 0)
		{
		for(i = collToRemove; i < numColls-1; i++)
		{
			siblCollecs[i].fname = siblCollecs[i+1].fname;
			siblCollecs[i].name = siblCollecs[i+1].name;
		}
		numColls--;

		// update combo box
		FreeChildren(IDC_COLLECTIONS);
		FreeChildren(IDC_CURRENTCOLL);
		for(i = 0; i < numColls; i++)
		{
			AddChild(IDC_COLLECTIONS, i, siblCollecs[i].name);
			AddChild(IDC_CURRENTCOLL, i, siblCollecs[i].name);
		}
		// set the menu to the current collection
		if(numColls == 0)								// do we have any collections now?
		{
			currentColl = -1;
			SetLong(IDC_COLLECTIONS, currentColl);
			SetLong(IDC_CURRENTCOLL, currentColl);
			ClearPresets();								// empty the loader of existing presets
			LayoutChanged(IDC_COLLECTIONS);
			LayoutChanged(IDC_CURRENTCOLL);
			SetLong(IDC_TABS, IDC_TAB1);
		}
		else
		{
			if(currentColl >= numColls)					// check for out of range
				currentColl = 0;
			SetLong(IDC_COLLECTIONS, currentColl);
			SetLong(IDC_CURRENTCOLL, currentColl);
			// set the collection folder
			if(filterFlag)
			{
				RemoveFilter(FALSE);
				ScanCollection(FALSE);
				ApplyFilter(TRUE);
			}
			else
				ScanCollection(TRUE);
		}
	}
}

Bool SIBLDialog::RemovesIBL(BaseDocument *doc, LONG renderEngine)
{
	GeData param;
	RenderData *rdata = NULL;
	BaseVideoPost *pvp = NULL;
	Bool foundVray = FALSE;
	BaseContainer *data = NULL;
	BaseList2D *op = NULL;
	
	// remove any sIBL preset from the scene, depending on render engine
	if(renderEngine == ADV_RENDER || renderEngine == VRAY_RENDER)
		RemoveExisting(doc);

	if(renderEngine == VRAY_RENDER && vrayFlag)
	{
		// get the render data and pvp objects
		rdata = doc->GetActiveRenderData();
		if(rdata)
		{
			pvp = rdata->GetFirstVideoPost();
			while(pvp)
			{
				if(pvp->GetType() == VPvray)
				{
					foundVray = TRUE;
					break;
				}
				pvp = pvp->GetNext();
			}
		}
		if(!foundVray)
			return FALSE;
		else		// we found Vray and have a valid pvp pointer
		{
			data = pvp->GetDataInstance();
			// first clear the environment background
			op = NULL;
			op = data->GetLink(VP_VRAYBRIDGE_ENVIRONMENT_BACKGROUNDTEX, doc);
			if(op)
				op->Remove();									// clear any shader from the environment background
			data->SetLink(VP_VRAYBRIDGE_ENVIRONMENT_BACKGROUNDTEX, NULL);
			param.SetReal(0.0);									// set rotation to zero
			pvp->SetParameter(DescID(VP_VRAYBRIDGE_ENVIRONMENT_BGMAPROTH), param, DESCFLAGS_SET_0);
			
			// now clear the GI texture
			op = NULL;
			op = data->GetLink(VP_VRAYBRIDGE_ENVIRONMENT_GIBACKGROUNDTEX, doc);
			if(op)
				op->Remove();									// clear any shader from the environment background
			data->SetLink(VP_VRAYBRIDGE_ENVIRONMENT_GIBACKGROUNDTEX, NULL);
			param.SetLong(0);									// turn off GI environment override
			pvp->SetParameter(DescID(VP_VRAYBRIDGE_ENVIRONMENT_BGNOGI), param, DESCFLAGS_SET_0);
			param.SetReal(0.0);									// set rotation to zero
			pvp->SetParameter(DescID(VP_VRAYBRIDGE_ENVIRONMENT_GIMAPROTH), param, DESCFLAGS_SET_0);
			param.SetReal(1.0);									// set intensity to one
			pvp->SetParameter(DescID(VP_VRAYBRIDGE_ENVIRONMENT_GIMAPMUL), param, DESCFLAGS_SET_0);
			
			// now the reflection texture
			op = NULL;
			op = data->GetLink(VP_VRAYBRIDGE_ENVIRONMENT_REFLBACKGROUNDTEX, doc);
			if(op)
				op->Remove();									// clear any shader from the environment background
			data->SetLink(VP_VRAYBRIDGE_ENVIRONMENT_REFLBACKGROUNDTEX, NULL);
			param.SetLong(0);									// turn off GI reflection override
			pvp->SetParameter(DescID(VP_VRAYBRIDGE_ENVIRONMENT_BGNOREFL), param, DESCFLAGS_SET_0);
			param.SetReal(0.0);									// set rotation to zero
			pvp->SetParameter(DescID(VP_VRAYBRIDGE_ENVIRONMENT_REFLMAPROTH), param, DESCFLAGS_SET_0);
			param.SetReal(1.0);									// set intensity to one
			pvp->SetParameter(DescID(VP_VRAYBRIDGE_ENVIRONMENT_REFLMAPMUL), param, DESCFLAGS_SET_0);
			
			// finally the refraction texture
			op = NULL;
			op = data->GetLink(VP_VRAYBRIDGE_ENVIRONMENT_REFRBACKGROUNDTEX, doc);
			if(op)
				op->Remove();									// clear any shader from the environment background
			data->SetLink(VP_VRAYBRIDGE_ENVIRONMENT_REFRBACKGROUNDTEX, NULL);
			param.SetLong(0);									// turn off GI reflection override
			pvp->SetParameter(DescID(VP_VRAYBRIDGE_ENVIRONMENT_BGNOREFR), param, DESCFLAGS_SET_0);
			param.SetReal(0.0);									// set rotation to zero
			pvp->SetParameter(DescID(VP_VRAYBRIDGE_ENVIRONMENT_REFRMAPROTH), param, DESCFLAGS_SET_0);
			param.SetReal(1.0);									// set intensity to one
			pvp->SetParameter(DescID(VP_VRAYBRIDGE_ENVIRONMENT_REFRMAPMUL), param, DESCFLAGS_SET_0);

			// send update message
			pvp->Message(MSG_UPDATE);
		}
	}

	return TRUE;
}

Bool SIBLDialog::ApplyPreset(void)
{
	// apply the current preset to the scene
	// return FALSE if some error (or there is no preset to apply), TRUE if all goes well
	BaseObject *baseNULL = NULL, *nameNULL = NULL, *warnNULL = NULL, *bgSky = NULL, *enSky = NULL, *reSky = NULL, *obj = NULL;
	Material *matSkyBG= NULL, *matSkyEN = NULL, *matSkyRE = NULL;
	BaseDocument *doc = NULL;
	GeData param;
	BaseTag *phongTag = NULL;

	if(presetIndex >= 0)			// belt and braces check for current preset
	{
		doc = GetActiveDocument();
		if(!doc)
			return FALSE;
		doc->StartUndo();
		// first turn off existing lights if the user wants that
		if(theOpts.turnOffLights)
		{
			obj = doc->GetFirstObject();
			if(obj)
				KillLights(doc, obj);
		}
		// if the user wants to overwrite an existing set, just delete it
		if(theOpts.overWriteExisting)
			RemoveExisting(doc);
		else
			DisableExisting(doc, doc->GetFirstObject());
		
		// create a set NULL
		baseNULL = BaseObject::Alloc(Onull);
		if(baseNULL)
		{
			baseNULL->SetName("sIBL_Set");
			doc->InsertObject(baseNULL, NULL, NULL, FALSE);
			doc->AddUndo(UNDOTYPE_NEW, baseNULL);
			// add name null object
			nameNULL = BaseObject::Alloc(Onull);
			if(nameNULL)
			{
				nameNULL->SetName(presets[presetIndex].name);
				doc->InsertObject(nameNULL, baseNULL, NULL, FALSE);
				doc->AddUndo(UNDOTYPE_NEW, nameNULL);
				// add warning null
				warnNULL = BaseObject::Alloc(Onull);
				if(warnNULL)
				{
					warnNULL->SetName("Important! Do not rename any of these objects!!");
					doc->InsertObject(warnNULL, baseNULL, NULL, FALSE);
					doc->AddUndo(UNDOTYPE_NEW, warnNULL);
				}
				// create Sky objects
				// add background sky object
				if(theOpts.loadBG && presets[presetIndex].numLS == 0)
				{
					if(!theOpts.useSpheres)
						bgSky = BaseObject::Alloc(Osky);
					else
					{
						// create spheres instead
						bgSky = BaseObject::Alloc(Osphere);
						param.SetReal(theOpts.sphereRadius);
						bgSky->SetParameter(DescID(PRIM_SPHERE_RAD), param, DESCFLAGS_SET_0);
						// add a phong tag
						phongTag = BaseTag::Alloc(Tphong);
						if(phongTag)
						{
							// turn on angle limit
							param.SetLong(1);
							phongTag->SetParameter(DescID(PHONGTAG_PHONG_ANGLELIMIT), param, DESCFLAGS_SET_0);
							bgSky->InsertTag(phongTag);
						}
					}
					if(bgSky)
					{
						bgSky->SetName("sIBL_BGSky");
						doc->InsertObject(bgSky, baseNULL, nameNULL, FALSE);
						doc->AddUndo(UNDOTYPE_NEW, bgSky);
					}
				}
				// add environment sky object
				if(theOpts.loadENV && presets[presetIndex].numLS == 0)
				{
					if(!theOpts.useSpheres)
						enSky = BaseObject::Alloc(Osky);
					else
					{
						// create spheres instead
						enSky = BaseObject::Alloc(Osphere);
						param.SetReal(theOpts.sphereRadius);
						enSky->SetParameter(DescID(PRIM_SPHERE_RAD), param, DESCFLAGS_SET_0);
					}
					if(enSky)
					{
						enSky->SetName("sIBL_ENSky");
						doc->InsertObject(enSky, baseNULL, nameNULL, FALSE);
						doc->AddUndo(UNDOTYPE_NEW, enSky);
					}
				}
				// add reflection sky object
				if(theOpts.loadREF && presets[presetIndex].numLS == 0)
				{
					if(!theOpts.useSpheres)
						reSky = BaseObject::Alloc(Osky);
					else
					{
						// create spheres instead
						reSky = BaseObject::Alloc(Osphere);
						param.SetReal(theOpts.sphereRadius);
						reSky->SetParameter(DescID(PRIM_SPHERE_RAD), param, DESCFLAGS_SET_0);
						// turn the 'render perfect' setting off
						param.SetLong(0);
						reSky->SetParameter(DescID(PRIM_SPHERE_PERFECT), param, DESCFLAGS_SET_0);
					}
					if(reSky)
					{
						reSky->SetName("sIBL_RESky");
						doc->InsertObject(reSky, baseNULL, nameNULL, FALSE);
						doc->AddUndo(UNDOTYPE_NEW, reSky);
					}
				}
			}

			// create mats & add textures, comp tags
			if(theOpts.loadBG && presets[presetIndex].bgfile != "")			// does the user want the background loaded, and if so is there a file?
			{
				matSkyBG = CreateSkyMat(doc, presets[presetIndex].bgfile, "sIBL_BGSky", SIBL_SKY_BG);
				if(bgSky && matSkyBG)										// do we have valid Sky object and material?
				{
					ApplySkyMat(doc, matSkyBG, bgSky);
				}
				if(bgSky)
				{
					CreateCompTag(doc, bgSky, SIBL_SKY_BG);					// add a comp tag if we have a valid sky object
				}
			}
			if(theOpts.loadENV && presets[presetIndex].evfile != "")		// same for the environment map
			{
				matSkyEN = CreateSkyMat(doc, presets[presetIndex].evfile, "sIBL_ENSky", SIBL_SKY_EN);
				if(enSky && matSkyEN)										// do we have valid Sky object and material?
				{
					ApplySkyMat(doc, matSkyEN, enSky);
				}
				if(enSky)
				{
					CreateCompTag(doc, enSky, SIBL_SKY_EN);					// add a comp tag if we have a valid sky object
				}
			}
			if(theOpts.loadREF && presets[presetIndex].reffile != "")
			{
				matSkyRE = CreateSkyMat(doc, presets[presetIndex].reffile, "sIBL_RESky", SIBL_SKY_RE);
				if(reSky && matSkyRE)										// do we have valid Sky object and material?
				{
					ApplySkyMat(doc, matSkyRE, reSky);
				}
				if(reSky)
				{
					CreateCompTag(doc, reSky, SIBL_SKY_RE);					// add a comp tag if we have a valid sky object
				}
			}

			// add a sun light (if applicable) and sun expression tag
			if(theOpts.loadSUN && presets[presetIndex].addsun)
			{
				CreateSun(doc, baseNULL, nameNULL);
			}

			// set up the GI - use default parameters
			if(theOpts.setupGI)
				SetupGI(doc);

			// set up any additional lights
			if(presets[presetIndex].numlights > 0 && theOpts.loadLights)
			{
				CreateLights(doc, baseNULL, nameNULL);
			}

			// set up any Lightsmiths
			if(presets[presetIndex].numLS > 0)
				CreateLS(doc, baseNULL, nameNULL);

			// align to north
			if(theOpts.alignNorth)
			{
				AlignNorth(doc, baseNULL);
			}
		}

		doc->EndUndo();
		EventAdd();

		return TRUE;
	}

	return FALSE;
}

Bool SIBLDialog::ApplyPresetVray(void)
{
	BaseDocument *doc;
	Bool success = FALSE;
	BaseObject *baseNULL = NULL, *nameNULL = NULL, *warnNULL = NULL, *obj = NULL;
	GeData param;

	if(!vrayFlag)			// safety check for Vray presence
	{
		MessageDialog("Vray does not appear to be installed on this machine. The render engine has not been changed.");
		success = FALSE;
	}
	else
	{
		if(presetIndex >= 0)			// belt and braces check for current preset
		{
			doc = GetActiveDocument();
			if(!doc)
				return FALSE;
			doc->StartUndo();
			success = SetupVray(doc);
			// first turn off existing lights if the user wants that
			if(theOpts.turnOffLights)
			{
				obj = doc->GetFirstObject();
				if(obj)
					KillLights(doc, obj);
			}
			// if the user wants to overwrite an existing set, just delete it
			if(theOpts.overWriteExisting)
				RemoveExisting(doc);
			else
				DisableExisting(doc, doc->GetFirstObject());
			
			// create a set NULL
			baseNULL = BaseObject::Alloc(Onull);
			if(baseNULL)
			{
				baseNULL->SetName("sIBL_Set");
				doc->InsertObject(baseNULL, NULL, NULL, FALSE);
				doc->AddUndo(UNDOTYPE_NEW, baseNULL);
				// add name null object
				nameNULL = BaseObject::Alloc(Onull);
				if(nameNULL)
				{
					nameNULL->SetName(presets[presetIndex].name);
					doc->InsertObject(nameNULL, baseNULL, NULL, FALSE);
					doc->AddUndo(UNDOTYPE_NEW, nameNULL);
					// add warning null
					warnNULL = BaseObject::Alloc(Onull);
					if(warnNULL)
					{
						warnNULL->SetName("Important! Do not rename any of these objects!!");
						doc->InsertObject(warnNULL, baseNULL, NULL, FALSE);
						doc->AddUndo(UNDOTYPE_NEW, warnNULL);
					}
				}
				// set the background and reflection images
				SetVrayEnv(doc, SIBL_SKY_BG, theOpts.loadBG);
				SetVrayEnv(doc, SIBL_SKY_RE, theOpts.loadREF);
				SetVrayEnv(doc, SIBL_SKY_EN, theOpts.loadENV);
				// does the user want a background sky?
				if(theOpts.vrayEditorBGSky && theOpts.loadBG)
				{
					CreateVrayBGSky(doc, baseNULL, nameNULL);
				}
				// add a dome light for the env HDRI if specified
				if(theOpts.loadENV && !theOpts.vrayUseGIEnv)
				{
					CreateVrayDomeLight(doc, baseNULL, nameNULL);
				}
				// add a sun light (if applicable) and Vray light tag
				if(theOpts.loadSUN && presets[presetIndex].addsun)
				{
					CreateSun(doc, baseNULL, nameNULL);
				}
				// set up any additional lights
				if(presets[presetIndex].numlights > 0 && theOpts.loadLights)
				{
					CreateLights(doc, baseNULL, nameNULL);
				}

				// set up any Lightsmiths
				if(presets[presetIndex].numLS > 0)
					CreateLS(doc, baseNULL, nameNULL);

				// align to north if required
				if(theOpts.alignNorth)
					AlignNorth(doc, baseNULL);
				// do we add a comp tag to a ground object?
				MakeVrayGroundMatte(doc);
				// show standard warning if the user wanted a Vray editor background
				if(theOpts.vrayEditorBGSky && theOpts.loadBG && theOpts.vrayWarnBG)
					MessageDialog("IMPORTANT: the editor background will not appear in the final render.\n\nPlease note that if you rotate the null object containing the background, the rendered image will no longer match what you see in the editor.\n\nTherefore, if you do rotate the null object, rememember to set the rotation of the images in the Vray environment settings to the same value!");
			}
			doc->EndUndo();
			EventAdd();
		}
	}

	return success;
}

Bool SIBLDialog::MakeVrayGroundMatte(BaseDocument *doc)
{
	LinkBoxGui *lbg;
	BaseObject *grndObj;
	BaseTag *comtag;
	GeData param;

	// try to find the linkbox control
	lbg = NULL;
	lbg = (LinkBoxGui*)FindCustomGui(IDC_VRAY_GROUNDOBJECT, 10009415);
	if(!lbg)
		GePrint("sIBL Loader: could not locate the linkbox gui.");
	else
	{
		grndObj = (BaseObject*)lbg->GetLink(doc);
		if(grndObj)
		{
			//GePrint("Ground object name is " + grndObj->GetName());
			comtag = BaseTag::Alloc(Vcomptag);
			if(comtag)
			{
				doc->AddUndo(UNDOTYPE_CHANGE, grndObj);
				grndObj->InsertTag(comtag);
				param.SetReal(0.0);					// receive GI to 0
				comtag->SetParameter(DescID(VRAYCOMPOSITINGTAG_RECGI), param, DESCFLAGS_SET_0);
				param.SetLong(1);					// matte surface on
				comtag->SetParameter(DescID(VRAYCOMPOSITINGTAG_MATTE), param, DESCFLAGS_SET_0);
				param.SetLong(1);					// shadows on
				comtag->SetParameter(DescID(VRAYCOMPOSITINGTAG_MATTESHADOWS), param, DESCFLAGS_SET_0);
				param.SetLong(0);					// turn off cast shadows
				comtag->SetParameter(DescID(VRAYCOMPOSITINGTAG_SHADOWVIS), param, DESCFLAGS_SET_0);
				param.SetLong(0);					// turn off generate GI
				comtag->SetParameter(DescID(VRAYCOMPOSITINGTAG_GIVIS), param, DESCFLAGS_SET_0);
				param.SetLong(0);					// turn off visible in reflections
				comtag->SetParameter(DescID(VRAYCOMPOSITINGTAG_REFLVIS), param, DESCFLAGS_SET_0);
				param.SetLong(0);					// turn off visible in refractions
				comtag->SetParameter(DescID(VRAYCOMPOSITINGTAG_REFRVIS), param, DESCFLAGS_SET_0);
				comtag->Message(MSG_UPDATE);
			}
		}
	}

	return TRUE;
}

Bool SIBLDialog::CreateVrayDomeLight(BaseDocument *doc, BaseObject *parent, BaseObject *prev)
{
	BaseObject *domeLight;
	GeData param;
	BaseTag *ltag = NULL;
	BaseContainer *data, *shddata = NULL, *tnfdata = NULL;
	BaseShader *shd = NULL, *tnf = NULL;

	// don't add light if there are Lightsmiths
	if(presets[presetIndex].numLS > 0)
		return TRUE;

	domeLight = BaseObject::Alloc(Olight);
	if(domeLight)
	{
		domeLight->SetName("sIBL_EnvLight");
		param.SetLong(LIGHT_TYPE_AREA);						// area light
		domeLight->SetParameter(DescID(LIGHT_TYPE), param, DESCFLAGS_SET_0);
		param.SetLong(2);									// hard shadows
		domeLight->SetParameter(DescID(LIGHT_SHADOWTYPE), param, DESCFLAGS_SET_0);
		if(theOpts.vrayUseSunIntensity)						// use sun intensity by default
			param.SetReal(presets[presetIndex].sunmulti);
		else												// otherwise use the environment multiplier
			param.SetReal(presets[presetIndex].evmulti);		// sun brightness
		domeLight->SetParameter(DescID(LIGHT_BRIGHTNESS), param, DESCFLAGS_SET_0);
		param.SetLong(LIGHT_AREADETAILS_SHAPE_HEMISPHERE);	// dome shape
		domeLight->SetParameter(DescID(LIGHT_AREADETAILS_SHAPE), param, DESCFLAGS_SET_0);
		param.SetReal(500.00);								// light dome radius
		domeLight->SetParameter(DescID(LIGHT_DETAILS_OUTERRADIUS), param, DESCFLAGS_SET_0);
		param.SetVector(presets[presetIndex].suncolor);
		domeLight->SetParameter(DescID(LIGHT_COLOR), param, DESCFLAGS_SET_0);
		doc->InsertObject(domeLight, parent, prev, FALSE);
		doc->AddUndo(UNDOTYPE_NEW, domeLight);
		// add a Vray light tag, which will take all the above values from the C4D light
		ltag = BaseTag::Alloc(Vlighttag);
		if(ltag)
		{
			doc->AddUndo(UNDOTYPE_CHANGE, domeLight);
			domeLight->InsertTag(ltag);
			param.SetLong(VRAYLIGHTTAG_AREA_TYPE_2);		// dome light
			ltag->SetParameter(DescID(VRAYLIGHTTAG_AREA_TYPE), param, DESCFLAGS_SET_0);
			param.SetLong(64);								// subdivs to 64
			ltag->SetParameter(DescID(VRAYLIGHTTAG_AREA_SUBDIVS), param, DESCFLAGS_SET_0);
			param.SetLong(1);								// use texture
			ltag->SetParameter(DescID(VRAYLIGHTTAG_AREA_DOME_USETEX), param, DESCFLAGS_SET_0);
			param.SetLong(VRAYLIGHTTAG_AREA_DOME_MAPTYPE_2);	// spherical mapping
			ltag->SetParameter(DescID(VRAYLIGHTTAG_AREA_DOME_MAPTYPE), param, DESCFLAGS_SET_0);
			param.SetLong(1);									// make invisible
			ltag->SetParameter(DescID(VRAYLIGHTTAG_AREA_INVISIBLE), param, DESCFLAGS_SET_0);
			param.SetLong(0);									// don't affect reflections
			ltag->SetParameter(DescID(VRAYLIGHTTAG_AREA_AFFECTREFL), param, DESCFLAGS_SET_0);
			data = ltag->GetDataInstance();
			shd = BaseShader::Alloc(Xbitmap);
			tnf = BaseShader::Alloc(Xbmflip);
			if(shd && tnf)
			{
				shddata = shd->GetDataInstance();
				tnfdata = tnf->GetDataInstance();
				// load the bitmap if it exists
				if(GeFExist(presets[presetIndex].path + Filename(presets[presetIndex].reffile), FALSE))
					shddata->SetFilename(BITMAPSHADER_FILENAME, presets[presetIndex].path + Filename(presets[presetIndex].reffile));
				// gamma correction for LWF
				shddata->SetLong(BITMAPSHADER_COLORPROFILE, BITMAPSHADER_COLORPROFILE_SRGB);
				tnfdata->SetLink(BMFLIPSHADER_TEXTURE, shd);
				param.SetLong(1);
				tnf->SetParameter(DescID(BMFLIPSHADER_FLIPX), param, DESCFLAGS_SET_0);
				data->SetLink(VRAYLIGHTTAG_AREA_DOME_TEX, tnf);
				shd->Message(MSG_UPDATE);
				tnf->InsertShader(shd);
				tnf->Message(MSG_UPDATE);
				doc->AddUndo(UNDOTYPE_NEW, ltag);
				ltag->InsertShader(tnf, NULL);
				doc->AddUndo(UNDOTYPE_NEW, ltag);
				ltag->Message(MSG_UPDATE);
			}
		}
		domeLight->Message(MSG_UPDATE);
	}
	return TRUE;
}

Bool SIBLDialog::SetVrayEnv(BaseDocument *doc, LONG envType, Bool enableEnv)
{
	BaseContainer *data, *shddata = NULL, *tnfdata = NULL, *bmdata = NULL;
	BaseShader *shd = NULL, *shd2 = NULL, *tnf = NULL, *tnf2 = NULL, *bm = NULL;
	GeData param;
	RenderData *rdata = NULL;
	BaseVideoPost *pvp = NULL;
	Bool foundVray = FALSE;
	Real rotX, gc;
	BaseList2D *op;

	// get the render data and pvp objects
	rdata = doc->GetActiveRenderData();
	if(rdata)
	{
		pvp = rdata->GetFirstVideoPost();
		while(pvp)
		{
			if(pvp->GetType() == VPvray)
			{
				foundVray = TRUE;
				break;
			}
			pvp = pvp->GetNext();
		}
		if(pvp)
			data = pvp->GetDataInstance();
	}
	if(!foundVray)
	{
		GePrint("Could not find the render data or the videopost for Vray.");
		return FALSE;
	}

	data = pvp->GetDataInstance();
	gc = 1.0;			// no gamma correction needed in Vray 1.2

	if(!enableEnv)						// turn the relevant sky dome off completely
	{
		if(envType == SIBL_SKY_BG)
		{
			op = NULL;
			op = data->GetLink(VP_VRAYBRIDGE_ENVIRONMENT_BACKGROUNDTEX, doc);
			if(op)
			{
				doc->AddUndo(UNDOTYPE_DELETE, pvp);
				op->Remove();									// clear any shader from the environment background
			}
			data->SetLink(VP_VRAYBRIDGE_ENVIRONMENT_BACKGROUNDTEX, NULL);
		}
		else if(envType == SIBL_SKY_RE)
		{
			op = NULL;
			op = data->GetLink(VP_VRAYBRIDGE_ENVIRONMENT_REFLBACKGROUNDTEX, doc);
			if(op)
			{
				doc->AddUndo(UNDOTYPE_DELETE, pvp);
				op->Remove();									// clear any shader from the environment background
			}
			data->SetLink(VP_VRAYBRIDGE_ENVIRONMENT_REFLBACKGROUNDTEX, NULL);
			param.SetLong(0);									// turn off reflection environment override
			doc->AddUndo(UNDOTYPE_CHANGE, pvp);
			pvp->SetParameter(DescID(VP_VRAYBRIDGE_ENVIRONMENT_BGNOREFL), param, DESCFLAGS_SET_0);
			op = NULL;
			op = data->GetLink(VP_VRAYBRIDGE_ENVIRONMENT_REFRBACKGROUNDTEX, doc);
			if(op)
			{
				doc->AddUndo(UNDOTYPE_DELETE, pvp);
				op->Remove();									// clear any shader from the environment background
			}
			data->SetLink(VP_VRAYBRIDGE_ENVIRONMENT_REFRBACKGROUNDTEX, NULL);
			param.SetLong(0);									// turn off refraction environment override
			doc->AddUndo(UNDOTYPE_CHANGE, pvp);
			pvp->SetParameter(DescID(VP_VRAYBRIDGE_ENVIRONMENT_BGNOREFR), param, DESCFLAGS_SET_0);
		}
		else if(envType == SIBL_SKY_EN)
		{
			op = NULL;
			op = data->GetLink(VP_VRAYBRIDGE_ENVIRONMENT_GIBACKGROUNDTEX, doc);
			if(op)
			{
				doc->AddUndo(UNDOTYPE_DELETE, pvp);
				op->Remove();									// clear any shader from the environment background
			}
			data->SetLink(VP_VRAYBRIDGE_ENVIRONMENT_GIBACKGROUNDTEX, NULL);
			param.SetLong(0);									// turn off GI environment override
			doc->AddUndo(UNDOTYPE_CHANGE, pvp);
			pvp->SetParameter(DescID(VP_VRAYBRIDGE_ENVIRONMENT_BGNOGI), param, DESCFLAGS_SET_0);
		}
		doc->AddUndo(UNDOTYPE_CHANGE, pvp);
		pvp->Message(MSG_UPDATE);
	}
	else
	{
		if(envType == SIBL_SKY_EN && !theOpts.vrayUseGIEnv)		// we want env lighting but with a dome light, so we'll turn the GI env off
		{
			op = NULL;
			op = data->GetLink(VP_VRAYBRIDGE_ENVIRONMENT_GIBACKGROUNDTEX, doc);
			if(op)
			{
				doc->AddUndo(UNDOTYPE_DELETE, pvp);
				op->Remove();									// clear any shader from the environment background
			}
			data->SetLink(VP_VRAYBRIDGE_ENVIRONMENT_GIBACKGROUNDTEX, NULL);		// remove shader
			param.SetLong(0);									// turn off GI environment override
			doc->AddUndo(UNDOTYPE_CHANGE, pvp);
			pvp->SetParameter(DescID(VP_VRAYBRIDGE_ENVIRONMENT_BGNOGI), param, DESCFLAGS_SET_0);
			pvp->Message(MSG_UPDATE);
		}
		else
		{
			rotX = ((presets[presetIndex].north - 0.25) * pi2) - pi;
			if(envType == SIBL_SKY_BG)
			{
				shd = BaseShader::Alloc(Xslafilter);
				bm = BaseShader::Alloc(Xbitmap);
			}
			else
				shd = BaseShader::Alloc(Xbitmap);
			tnf = BaseShader::Alloc(Xbmflip);
			if(shd && tnf)
			{
				shddata = shd->GetDataInstance();	// the Xslafilter shader for BG, or the Xbitmap shader for the others
				tnfdata = tnf->GetDataInstance();
				if(envType == SIBL_SKY_BG && bm && presets[presetIndex].bgfile != "")
				{
					bmdata = bm->GetDataInstance();
					// load the bitmap if it exists
					if(GeFExist(presets[presetIndex].path + Filename(presets[presetIndex].bgfile), FALSE))
						bmdata->SetFilename(BITMAPSHADER_FILENAME, presets[presetIndex].path + Filename(presets[presetIndex].bgfile));
					bmdata->SetLong(BITMAPSHADER_COLORPROFILE, BITMAPSHADER_COLORPROFILE_SRGB);
					param.SetReal(0.0);				// zero any rotation on X
					doc->AddUndo(UNDOTYPE_CHANGE, pvp);
					pvp->SetParameter(DescID(VP_VRAYBRIDGE_ENVIRONMENT_BGMAPROTH), param, DESCFLAGS_SET_0);
					if(theOpts.alignNorth)
					{
						param.SetReal(rotX);
						doc->AddUndo(UNDOTYPE_CHANGE, pvp);
						pvp->SetParameter(DescID(VP_VRAYBRIDGE_ENVIRONMENT_BGMAPROTH), param, DESCFLAGS_SET_0);
					}
					shddata->SetLink(SLA_FILTER_TEXTURE, bm);
					if(theOpts.vrayLWFMode == VRAY_LWF_32BIT)				// set the gamma according to the LWF mode
						param.SetReal(0.4545);
					else
						param.SetReal(1.0);
					shd->SetParameter(DescID(SLA_FILTER_GAMMA), param, DESCFLAGS_SET_0);
					tnfdata->SetLink(BMFLIPSHADER_TEXTURE, shd);
					param.SetLong(1);				// flip bitmap on X axis
					tnf->SetParameter(DescID(BMFLIPSHADER_FLIPX), param, DESCFLAGS_SET_0);
					doc->AddUndo(UNDOTYPE_CHANGE, pvp);
					data->SetLink(VP_VRAYBRIDGE_ENVIRONMENT_BACKGROUNDTEX, tnf);
				}
				else if(envType == SIBL_SKY_RE && presets[presetIndex].reffile != "")
				{
					param.SetLong(1);				// turn on background reflection override
					doc->AddUndo(UNDOTYPE_CHANGE, pvp);
					pvp->SetParameter(DescID(VP_VRAYBRIDGE_ENVIRONMENT_BGNOREFL), param, DESCFLAGS_SET_0);
					if(GeFExist(presets[presetIndex].path + Filename(presets[presetIndex].reffile), FALSE))
					{
						shddata->SetFilename(BITMAPSHADER_FILENAME, presets[presetIndex].path + Filename(presets[presetIndex].reffile));
						shddata->SetLong(BITMAPSHADER_COLORPROFILE, BITMAPSHADER_COLORPROFILE_SRGB);
						//shddata->SetReal(BITMAPSHADER_GAMMA, presets[presetIndex].refgamma / gc);
						param.SetReal(0.0);				// zero any rotation on X
						doc->AddUndo(UNDOTYPE_CHANGE, pvp);
						pvp->SetParameter(DescID(VP_VRAYBRIDGE_ENVIRONMENT_REFLMAPROTH), param, DESCFLAGS_SET_0);
						if(theOpts.alignNorth)
						{
							param.SetReal(rotX);
							doc->AddUndo(UNDOTYPE_CHANGE, pvp);
							pvp->SetParameter(DescID(VP_VRAYBRIDGE_ENVIRONMENT_REFLMAPROTH), param, DESCFLAGS_SET_0);
						}
						param.SetReal(presets[presetIndex].refmulti);
						doc->AddUndo(UNDOTYPE_CHANGE, pvp);
						pvp->SetParameter(DescID(VP_VRAYBRIDGE_ENVIRONMENT_REFLMAPMUL), param, DESCFLAGS_SET_0);
					}
					tnfdata->SetLink(BMFLIPSHADER_TEXTURE, shd);
					param.SetLong(1);
					tnf->SetParameter(DescID(BMFLIPSHADER_FLIPX), param, DESCFLAGS_SET_0);
					doc->AddUndo(UNDOTYPE_CHANGE, pvp);
					data->SetLink(VP_VRAYBRIDGE_ENVIRONMENT_REFLBACKGROUNDTEX, tnf);
					// add refraction override if user wants this
					if(theOpts.vrayAddRefraction)
					{
						shd2 = BaseShader::Alloc(Xbitmap);
						tnf2 = BaseShader::Alloc(Xbmflip);
						if(shd2 && tnf2)
						{
							shddata = shd2->GetDataInstance();	// the Xbitmap shader
							tnfdata = tnf2->GetDataInstance();
							param.SetLong(1);					// turn on refraction environment override
							doc->AddUndo(UNDOTYPE_CHANGE, pvp);
							pvp->SetParameter(DescID(VP_VRAYBRIDGE_ENVIRONMENT_BGNOREFR), param, DESCFLAGS_SET_0);
							if(GeFExist(presets[presetIndex].path + Filename(presets[presetIndex].reffile), FALSE))
							{
								shddata->SetFilename(BITMAPSHADER_FILENAME, presets[presetIndex].path + Filename(presets[presetIndex].reffile));
								shddata->SetLong(BITMAPSHADER_COLORPROFILE, BITMAPSHADER_COLORPROFILE_SRGB);
								param.SetReal(0.0);				// zero any rotation on X
								doc->AddUndo(UNDOTYPE_CHANGE, pvp);
								pvp->SetParameter(DescID(VP_VRAYBRIDGE_ENVIRONMENT_REFRMAPROTH), param, DESCFLAGS_SET_0);
								if(theOpts.alignNorth)
								{
									param.SetReal(rotX);
									doc->AddUndo(UNDOTYPE_CHANGE, pvp);
									pvp->SetParameter(DescID(VP_VRAYBRIDGE_ENVIRONMENT_REFRMAPROTH), param, DESCFLAGS_SET_0);
								}
								param.SetReal(presets[presetIndex].refmulti);
								doc->AddUndo(UNDOTYPE_CHANGE, pvp);
								pvp->SetParameter(DescID(VP_VRAYBRIDGE_ENVIRONMENT_REFRMAPMUL), param, DESCFLAGS_SET_0);
							}
							tnfdata->SetLink(BMFLIPSHADER_TEXTURE, shd2);
							param.SetLong(1);
							tnf2->SetParameter(DescID(BMFLIPSHADER_FLIPX), param, DESCFLAGS_SET_0);
							doc->AddUndo(UNDOTYPE_CHANGE, pvp);
							data->SetLink(VP_VRAYBRIDGE_ENVIRONMENT_REFRBACKGROUNDTEX, tnf2);
						}
					}
					else
					{
						param.SetLong(0);				// turn off refraction environment override
						doc->AddUndo(UNDOTYPE_CHANGE, pvp);
						pvp->SetParameter(DescID(VP_VRAYBRIDGE_ENVIRONMENT_BGNOREFR), param, DESCFLAGS_SET_0);
					}
				}
				else if(envType == SIBL_SKY_EN && presets[presetIndex].evfile != "" && theOpts.vrayUseGIEnv)
				{
					param.SetLong(1);									// turn on GI environment override
					doc->AddUndo(UNDOTYPE_CHANGE, pvp);
					pvp->SetParameter(DescID(VP_VRAYBRIDGE_ENVIRONMENT_BGNOGI), param, DESCFLAGS_SET_0);
					if(GeFExist(presets[presetIndex].path + Filename(presets[presetIndex].evfile), FALSE))
					{
						shddata->SetFilename(BITMAPSHADER_FILENAME, presets[presetIndex].path + Filename(presets[presetIndex].evfile));
						shddata->SetLong(BITMAPSHADER_COLORPROFILE, BITMAPSHADER_COLORPROFILE_SRGB);
						param.SetReal(0.0);				// zero any rotation on X
						doc->AddUndo(UNDOTYPE_CHANGE, pvp);
						pvp->SetParameter(DescID(VP_VRAYBRIDGE_ENVIRONMENT_GIMAPROTH), param, DESCFLAGS_SET_0);
						if(theOpts.alignNorth)
						{
							param.SetReal(rotX);
							doc->AddUndo(UNDOTYPE_CHANGE, pvp);
							pvp->SetParameter(DescID(VP_VRAYBRIDGE_ENVIRONMENT_GIMAPROTH), param, DESCFLAGS_SET_0);
						}
						param.SetReal(presets[presetIndex].evmulti);
						doc->AddUndo(UNDOTYPE_CHANGE, pvp);
						pvp->SetParameter(DescID(VP_VRAYBRIDGE_ENVIRONMENT_GIMAPMUL), param, DESCFLAGS_SET_0);
					}
					tnfdata->SetLink(BMFLIPSHADER_TEXTURE, shd);
					param.SetLong(1);
					tnf->SetParameter(DescID(BMFLIPSHADER_FLIPX), param, DESCFLAGS_SET_0);
					doc->AddUndo(UNDOTYPE_CHANGE, pvp);
					data->SetLink(VP_VRAYBRIDGE_ENVIRONMENT_GIBACKGROUNDTEX, tnf);
				}
				if(bm)
				{
					bm->Message(MSG_UPDATE);
					shd->InsertShader(bm, NULL);
				}
				shd->Message(MSG_UPDATE);
				tnf->InsertShader(shd, NULL);
				tnf->Message(MSG_UPDATE);
				pvp->InsertShader(tnf, NULL);
				doc->AddUndo(UNDOTYPE_NEW, pvp);
				if(shd2 && tnf2)
				{
					shd2->Message(MSG_UPDATE);
					tnf2->InsertShader(shd2, NULL);
					tnf2->Message(MSG_UPDATE);
					pvp->InsertShader(tnf2, NULL);
					doc->AddUndo(UNDOTYPE_NEW, pvp);
				}
				pvp->Message(MSG_UPDATE);
			}
			else
				GePrint("Could not allocate the shaders for the Vray environment.");
		}
	}
	return TRUE;
}

Bool SIBLDialog::SetupVray(BaseDocument *doc)
{
	RenderData *rdata = NULL;
	BaseVideoPost *pvp = NULL, *dga = NULL;
	Bool foundVray = FALSE, foundDeGamma = FALSE;
	BaseContainer bc;
	Bool success;
	GeData param;

	rdata = doc->GetActiveRenderData();
	if(rdata)
	{
		bc = rdata->GetData();
		// check to see if Vray is the render engine
		if(bc.GetLong(RDATA_RENDERENGINE) != VPvray)
		{
			bc.SetLong(RDATA_RENDERENGINE, VPvray);
			doc->AddUndo(UNDOTYPE_CHANGE, rdata);
			rdata->SetData(bc);
		}

		// turn off DeGamma as it causes problems with Vray
		dga = rdata->GetFirstVideoPost();
		while(dga)
		{
			if(dga->GetType() == SIBL_DEGAMMA)
			{
				foundDeGamma = TRUE;
				break;
			}
			dga = dga->GetNext();
		}
		if(foundDeGamma)
		{
			doc->AddUndo(UNDOTYPE_CHANGE, dga);
			dga->SetBit(BIT_VPDISABLED);
		}

		// now check if the bridge is in the videopost list
		pvp = rdata->GetFirstVideoPost();
		while(pvp)
		{
			if(pvp->GetType() == VPvray)
			{
				foundVray = TRUE;
				// turn the VP on if it's disabled
				if(pvp->GetBit(BIT_VPDISABLED))
				{
					doc->AddUndo(UNDOTYPE_CHANGE, pvp);
					pvp->ToggleBit(BIT_VPDISABLED);
				}
				break;
			}
			pvp = pvp->GetNext();
		}
		if(!foundVray)				// couldn't find the bridge, so add it to the VP list
		{
			pvp = BaseVideoPost::Alloc(VPvray);
			if(pvp)
			{
				foundVray = TRUE;
				rdata->InsertVideoPost(pvp, NULL);
				doc->AddUndo(UNDOTYPE_NEW, pvp);
			}
		}
		if(pvp && foundVray)
		{
			// turn off Affect Background
			param.SetLong(0);
			doc->AddUndo(UNDOTYPE_CHANGE, pvp);
			pvp->SetParameter(DescID(VP_VRAYBRIDGE_COLORMAPPING_AFFECTBG), param, DESCFLAGS_SET_0);
			// set up GI if the user desires it
			if(theOpts.setupGI)
			{
				param.SetLong(1);
				doc->AddUndo(UNDOTYPE_CHANGE, pvp);
				pvp->SetParameter(DescID(VP_VRAYBRIDGE_GI_ON), param, DESCFLAGS_SET_0);
			}
			// set up linear workflow if required
			if(theOpts.vrayLWFMode == VRAY_LWF_OFF)
			{
				param.SetLong(VP_VRAYBRIDGE_COLORMAPPING_TYPE_6);		// Reinhard
				doc->AddUndo(UNDOTYPE_CHANGE, pvp);
				pvp->SetParameter(DescID(VP_VRAYBRIDGE_COLORMAPPING_TYPE), param, DESCFLAGS_SET_0);
				param.SetReal(1.0);										// multiplier = 1
				doc->AddUndo(UNDOTYPE_CHANGE, pvp);
				pvp->SetParameter(DescID(VP_VRAYBRIDGE_COLORMAPPING_DARKMUL), param, DESCFLAGS_SET_0);
				param.SetReal(1.0);										// burn = 1.0
				doc->AddUndo(UNDOTYPE_CHANGE, pvp);
				pvp->SetParameter(DescID(VP_VRAYBRIDGE_COLORMAPPING_BRIGHTMUL), param, DESCFLAGS_SET_0);
				param.SetReal(1.0);										// gamma = 1.0
				doc->AddUndo(UNDOTYPE_CHANGE, pvp);
				pvp->SetParameter(DescID(VP_VRAYBRIDGE_COLORMAPPING_GAMMA), param, DESCFLAGS_SET_0);
				param.SetLong(0);										// LWF off
				doc->AddUndo(UNDOTYPE_CHANGE, pvp);
				pvp->SetParameter(DescID(VP_VRAYBRIDGE_COLORMAPPING_LWF), param, DESCFLAGS_SET_0);
				bc.SetLong(RDATA_FORMATDEPTH, RDATA_FORMATDEPTH_8);			// 8-bit format
				doc->AddUndo(UNDOTYPE_CHANGE, rdata);
				rdata->SetData(bc);
			}
			else if(theOpts.vrayLWFMode == VRAY_LWF_8BIT || theOpts.vrayLWFMode == VRAY_LWF_32BIT)
			{
				// will apply the LWF settings, either those stored previously or the default settings
				param.SetLong(VP_VRAYBRIDGE_COLORMAPPING_TYPE_0);											// color mapping type = linear multiply
				doc->AddUndo(UNDOTYPE_CHANGE, pvp);
				pvp->SetParameter(DescID(VP_VRAYBRIDGE_COLORMAPPING_TYPE), param, DESCFLAGS_SET_0);
				param.SetReal(1.0);										// multiplier
				doc->AddUndo(UNDOTYPE_CHANGE, pvp);
				pvp->SetParameter(DescID(VP_VRAYBRIDGE_COLORMAPPING_DARKMUL), param, DESCFLAGS_SET_0);
				param.SetReal(1.0);										// burn
				doc->AddUndo(UNDOTYPE_CHANGE, pvp);
				pvp->SetParameter(DescID(VP_VRAYBRIDGE_COLORMAPPING_BRIGHTMUL), param, DESCFLAGS_SET_0);
				param.SetReal(2.2);										// gamma
				doc->AddUndo(UNDOTYPE_CHANGE, pvp);
				pvp->SetParameter(DescID(VP_VRAYBRIDGE_COLORMAPPING_GAMMA), param, DESCFLAGS_SET_0);
				param.SetLong(1);										// LWF on
				doc->AddUndo(UNDOTYPE_CHANGE, pvp);
				pvp->SetParameter(DescID(VP_VRAYBRIDGE_COLORMAPPING_LWF), param, DESCFLAGS_SET_0);
				if(theOpts.vrayLWFMode == VRAY_LWF_8BIT)
				{
					param.SetLong(0);											// turn off adaptation only for 8-bit
					doc->AddUndo(UNDOTYPE_CHANGE, pvp);
					pvp->SetParameter(DescID(VP_VRAYBRIDGE_COLORMAPPING_ADAPTATION), param, DESCFLAGS_SET_0);
					bc.SetLong(RDATA_FORMATDEPTH, RDATA_FORMATDEPTH_8);			// 8-bit format
					doc->AddUndo(UNDOTYPE_CHANGE, rdata);
					rdata->SetData(bc);
				}
				else if(theOpts.vrayLWFMode == VRAY_LWF_32BIT)
				{
					param.SetLong(1);											// turn on adaptation only for 32-bit
					doc->AddUndo(UNDOTYPE_CHANGE, pvp);
					pvp->SetParameter(DescID(VP_VRAYBRIDGE_COLORMAPPING_ADAPTATION), param, DESCFLAGS_SET_0);
					bc.SetLong(RDATA_FORMATDEPTH, RDATA_FORMATDEPTH_32);			// 32-bit format
					doc->AddUndo(UNDOTYPE_CHANGE, rdata);
					rdata->SetData(bc);
				}
			}
			rdata->Message(MSG_UPDATE);
		}
		success = TRUE;
	}
	else
	{
		GePrint("sIBL Loader: Error - could not obtain the active render data to set up Vray.");
		success = FALSE;
	}

	return success;
}

void SIBLDialog::RemoveExisting(BaseDocument *doc)
{
	// delete an existing sIBL set and all its children
	BaseObject *obj = NULL;
	BaseMaterial *mat = NULL;
	Bool foundMat;

	// find the set null
	do 
	{
		obj = doc->SearchObject("sIBL_Set");
		if(obj)				// there is an existing set
		{
			obj->Remove();
			DeleteObject(doc, obj, TRUE);
		}
	} while (obj);

	// get rid of the materials
	do 
	{
		mat = doc->SearchMaterial("sIBL_BGSky");
		if(mat)
		{
			foundMat = TRUE;
			mat->Remove();
			BaseMaterial::Free(mat);
		}
		else
			foundMat = FALSE;
	} while (foundMat);
	mat = NULL;
	do 
	{
		mat = doc->SearchMaterial("sIBL_ENSky");
		if(mat)
		{
			foundMat = TRUE;
			mat->Remove();
			BaseMaterial::Free(mat);
		}
		else
			foundMat = FALSE;
	} while (foundMat);
	mat = NULL;
	do 
	{
		mat = doc->SearchMaterial("sIBL_RESky");
		if(mat)
		{
			foundMat = TRUE;
			mat->Remove();
			BaseMaterial::Free(mat);
		}
		else
			foundMat = FALSE;
	} while (foundMat);
	mat = NULL;
	do 
	{
		mat = doc->SearchMaterial("sIBL_Vray_editorBG");
		if(mat)
		{
			foundMat = TRUE;
			mat->Remove();
			BaseMaterial::Free(mat);
		}
		else
			foundMat = FALSE;
	} while (foundMat);
	mat = NULL;
	do 
	{
		mat = doc->SearchMaterial("sIBL LS illum.");
		if(mat)
		{
			foundMat = TRUE;
			mat->Remove();
			BaseMaterial::Free(mat);
		}
		else
			foundMat = FALSE;
	} while (foundMat);
	mat = NULL;
	do 
	{
		mat = doc->SearchMaterial("sIBL LS frame");
		if(mat)
		{
			foundMat = TRUE;
			mat->Remove();
			BaseMaterial::Free(mat);
		}
		else
			foundMat = FALSE;
	} while (foundMat);
}


void SIBLDialog::DisableExisting(BaseDocument *doc, BaseObject *obj)
{
	BaseObject *next = NULL;

	if(!obj)			// if there is no object, return
		return;

	// is this object an sIBL set?
	if(obj->GetName() == "sIBL_Set")
	{
		doc->AddUndo(UNDOTYPE_CHANGE, obj);
		obj->SetEditorMode(MODE_OFF);
		doc->AddUndo(UNDOTYPE_CHANGE, obj);
		obj->SetRenderMode(MODE_OFF);
		obj->Message(MSG_UPDATE);
	}
	else			// no, so check children
	{
		next = obj->GetDown();
		if(next)
			DisableExisting(doc, next);
	}
	// check siblings
	next = obj->GetNext();
	if(next)
		DisableExisting(doc, next);
}

void SIBLDialog::DeleteObject(BaseDocument *doc, BaseObject *obj, Bool isRoot)
{
	BaseObject *next = NULL;

	// does the object have children
	do
	{
		next = obj->GetDown();
		if(next)
			DeleteObject(doc, next, FALSE);
	} while(next);

	// delete siblings, but only if not the root object
	if (!isRoot)
	{
		next = obj->GetNext();
		if(next)
			DeleteObject(doc, next, FALSE);
	}

	// finally delete this object
	obj->Remove();
	BaseObject::Free(obj);

}


void SIBLDialog::KillLights(BaseDocument *doc, BaseObject *obj)
{
	// kill lights by recursive search
	BaseObject *next = NULL;

	//first, is obj a light itself?
	if(obj->GetType() == Olight)
	{
		doc->AddUndo(UNDOTYPE_CHANGE, obj);
		obj->SetDeformMode(FALSE);		// turn off light
	}
	// next, does it have any child objects?
	next = obj->GetDown();
	if(next)
	{
		KillLights(doc, next);
	}
	// finally, does it have any siblings?
	next = obj->GetNext();
	if(next)
		KillLights(doc, next);
}


void SIBLDialog::AlignNorth(BaseDocument *doc, BaseObject *parent)
{
	// align the entire setup to point Z-axis to the image north
	Vector rot;

	// don't do this for Lightsmith setups
	if(presets[presetIndex].numLS > 0)
		return;

	rot.x = ((presets[presetIndex].north - 0.25) * pi2) - pi;
	rot.y = 0.0;
	rot.z = 0.0;
	doc->AddUndo(UNDOTYPE_CHANGE, parent);
	parent->SetAbsRot(rot);
	parent->Message(MSG_UPDATE);
}

Bool SIBLDialog::CreateLights(BaseDocument *doc, BaseObject *parent, BaseObject *prev)
{
	BaseObject *light;
	Bool success = FALSE;
	GeData param;
	Vector lightRotPos;
	Real theta;
	LONG i;
	lightData *lData;
	BaseTag *ltag;

	lData = presets[presetIndex].lights;			// pointer to the light data
	if(lData)										// pointer check
	{
		for(i = 0; i < presets[presetIndex].numlights; i++)
		{
			light = BaseObject::Alloc(Olight);
			if(light)
			{
				light->SetName(lData[i].lightname);
				if(theOpts.lightType == SPOTLIGHT)
					param.SetLong(LIGHT_TYPE_SPOT);								// set to spot light
				else
					param.SetLong(LIGHT_TYPE_OMNI);
				light->SetParameter(DescID(LIGHT_TYPE), param, DESCFLAGS_SET_0);
				param.SetReal(lData[i].lightmulti);							// set light brightness
				light->SetParameter(DescID(LIGHT_BRIGHTNESS), param, DESCFLAGS_SET_0);
				param.SetVector(lData[i].lightcolor);						// set light color
				light->SetParameter(DescID(LIGHT_COLOR), param, DESCFLAGS_SET_0);
				doc->InsertObject(light, parent, prev, FALSE);
				doc->AddUndo(UNDOTYPE_NEW, light);
				// are we using Vray?
				if(theOpts.renderEngine == VRAY_RENDER && vrayFlag)
				{
					// add a Vray light tag, which will take all the above values from the C4D light
					ltag = BaseTag::Alloc(Vlighttag);
					if(ltag)
					{
						doc->AddUndo(UNDOTYPE_CHANGE, light);
						light->InsertTag(ltag);
					}
				}
				// set light rotation if this is a spotlight
				theta = (pi05 - (lData[i].lightv * pi)) * -1;
				if(theOpts.lightType == SPOTLIGHT)
				{
					lightRotPos.x = (((lData[i].lightu + 0.25) * pi2) * -1.0) + Rad(180.0);			// correction of *-1.0 and +180 degrees is to allow inversion of texture tag
					lightRotPos.y = theta;
					lightRotPos.z = 0.0;
					doc->AddUndo(UNDOTYPE_CHANGE, light);
					light->SetAbsRot(lightRotPos);
				}
				// and change to an arbitrary position for user convenience
				lightRotPos.x = 0.0;
				lightRotPos.y = 500.00 * Sin(theta) * -1.0;
				lightRotPos.z = 0.0;
				doc->AddUndo(UNDOTYPE_CHANGE, light);
				light->SetAbsPos(lightRotPos);
				light->Message(MSG_UPDATE);
				success = TRUE;
			}
		}
	}

	return success;
}

Bool SIBLDialog::CreateLS(BaseDocument *doc, BaseObject *parent, BaseObject *prev)
{
	Bool success = FALSE;
	LONG i;
	lsData *lsD;
	BaseObject *holder = NULL, *obj = NULL;
	Filename lsobj;
	BaseDocument *newdoc;
	String name;
	BaseTag *ltag = NULL;

	lsD = presets[presetIndex].ls;			// pointer to the light data
	if(lsD)										// pointer check
	{
		for(i = 0; i < presets[presetIndex].numLS; i++)
		{
			holder = BaseObject::Alloc(Onull);
			if(holder)
			{
				doc->InsertObject(holder, parent, prev);
				holder->SetName("Lightsmith objects");
				lsobj = presets[presetIndex].path + Filename(lsD[i].lsobjectfile);
				newdoc = LoadDocument(lsobj, SCENEFILTER_OBJECTS, NULL);
				if(newdoc)
				{
					obj = newdoc->GetFirstObject();
					if(obj)
					{
						obj->Remove();
						doc->InsertObject(obj, holder, NULL);
						lsobj = Filename(lsD[i].lsobjectfile);
						lsobj.ClearSuffix();
						obj->SetName(lsobj.GetFileString());
						obj->SetAbsScale(theOpts.ls_scalemulti * lsD[i].lsmulti);
						if(theOpts.renderEngine == ADV_RENDER || (theOpts.renderEngine == VRAY_RENDER && vrayFlag && theOpts.vrayLSMode != VRAY_LS_MODE_MESH))
							CreateLSMat(doc, obj, LS_ILLUM, lsD[i]);
						else if(theOpts.renderEngine == VRAY_RENDER && vrayFlag && theOpts.vrayLSMode == VRAY_LS_MODE_MESH)
							ltag = CreateMeshLight(doc, obj, lsD[i], holder, NULL);
						obj->Message(MSG_UPDATE);
						KillDocument(newdoc);
					}
				}
				lsobj = presets[presetIndex].path + Filename(lsD[i].lsframefile);
				newdoc = LoadDocument(lsobj, SCENEFILTER_OBJECTS, NULL);
				if(newdoc)
				{
					obj = newdoc->GetFirstObject();
					if(obj)
					{
						obj->Remove();
						doc->InsertObject(obj, holder, NULL);
						lsobj = Filename(lsD[i].lsframefile);
						lsobj.ClearSuffix();
						obj->SetName(lsobj.GetFileString());
						obj->SetAbsScale(theOpts.ls_scalemulti * lsD[i].lsmulti);
						CreateLSMat(doc, obj, LS_FRAME, lsD[i]);
						obj->Message(MSG_UPDATE);
						KillDocument(newdoc);
					}
				}
				EventAdd(EVENT_0);
			}
		}
	}

	return success;
}

BaseTag* SIBLDialog::CreateMeshLight(BaseDocument *doc, BaseObject *obj, lsData lsd, BaseObject *parent, BaseObject *prev)
{
	BaseObject *meshLight;
	GeData param;
	BaseTag *ltag = NULL;
	BaseContainer *data;
	AutoAlloc<BaseLink>lk;

	meshLight = BaseObject::Alloc(Olight);
	if(meshLight)
	{
		doc->InsertObject(meshLight, parent, prev, FALSE);
		doc->AddUndo(UNDOTYPE_NEW, meshLight);
		meshLight->SetName("Lightsmith mesh light");

		param.SetLong(LIGHT_TYPE_AREA);						// area light
		meshLight->SetParameter(DescID(LIGHT_TYPE), param, DESCFLAGS_SET_0);
		param.SetVector(lsd.lscolor);
		meshLight->SetParameter(DescID(LIGHT_COLOR), param, DESCFLAGS_SET_0);
		// add a Vray light tag, which will take all the above values from the C4D light
		ltag = BaseTag::Alloc(Vlighttag);
		if(ltag)
		{
			doc->AddUndo(UNDOTYPE_CHANGE, meshLight);
			meshLight->InsertTag(ltag);
			meshLight->Message(MSG_UPDATE);
			EventAdd(EVENT_0);
			data = ltag->GetDataInstance();
			data->SetLong(VRAYLIGHTTAG_COMMON_LIGHTTYPE, 3);
			data->SetLong(VRAYLIGHTTAG_AREA_TYPE, VRAYLIGHTTAG_AREA_TYPE_3);
			data->SetLink(VRAYLIGHTTAG_AREA_MESH_GEOMETRY, obj);
			data->SetLong(VRAYLIGHTTAG_AREA_SUBDIVS, 64);
			ltag->Message(MSG_UPDATE);
		}
	}
	return ltag;
}

void SIBLDialog::CreateLSMat(BaseDocument *doc, BaseObject *obj, LONG lstype, lsData lsD)
{
	BaseMaterial *bm = NULL;
	Material *lsMat = NULL;
	BaseContainer *data, *shddata;
	BaseShader *shd = NULL, *shd2 = NULL;
	GeData param;
	TextureTag *texTag;

	if(theOpts.renderEngine == VRAY_RENDER && vrayFlag)
	{
		bm = BaseMaterial::Alloc(ID_VRAYBRDF_MATERIAL);
		lsMat = static_cast<Material*>(bm);
	}
	else
		lsMat = Material::Alloc();
	if(lsMat)
	{
		data = lsMat->GetDataInstance();
		// environment map into luminance channel, others into color channel
		if(lstype == LS_ILLUM)
		{
			lsMat->SetName("sIBL LS illum.");
			if(theOpts.renderEngine == VRAY_RENDER && vrayFlag)
			{
				data->SetBool(VRAYMATERIAL_USE_LUMINANCE, TRUE);
				data->SetReal(VRAYMATERIAL_LUMINANCE_TRANSP_MULT, 0.0);
				data->SetBool(VRAYMATERIAL_USE_COLOR, TRUE);
				if(lsD.lsmaskfile != "")
					data->SetBool(VRAYMATERIAL_USE_TRANSP, TRUE);			// enable alpha if there is a mask file
			}
			else
			{
				lsMat->SetChannelState(CHANNEL_COLOR, TRUE);
				lsMat->SetChannelState(CHANNEL_LUMINANCE, TRUE);
				lsMat->SetChannelState(CHANNEL_SPECULAR, FALSE);
				if(lsD.lsmaskfile != "")
					lsMat->SetChannelState(CHANNEL_ALPHA, TRUE);			// enable alpha if there is a mask file
			}
		}
		else if(lstype == LS_FRAME)
		{
			lsMat->SetName("sIBL LS frame");
			if(theOpts.renderEngine == VRAY_RENDER && vrayFlag)
			{
				data->SetBool(VRAYMATERIAL_USE_COLOR, TRUE);
			}
			else
			{
				lsMat->SetChannelState(CHANNEL_COLOR, TRUE);
				lsMat->SetChannelState(CHANNEL_SPECULAR, TRUE);
				lsMat->SetChannelState(CHANNEL_LUMINANCE, FALSE);
			}
		}
		else
			return;

		// set texture map
		if(lstype == LS_ILLUM)
		{
			shd = BaseShader::Alloc(Xbitmap);
			if(shd)
			{
				// load the bitmap if it exists
				if(GeFExist(presets[presetIndex].path + Filename(lsD.lsfile), FALSE))
				{
					shddata = shd->GetDataInstance();
					shddata->SetFilename(BITMAPSHADER_FILENAME, presets[presetIndex].path + Filename(lsD.lsfile));
					shd->Message(MSG_UPDATE);
					shddata->SetLong(BITMAPSHADER_COLORPROFILE, BITMAPSHADER_COLORPROFILE_SRGB);
					if(theOpts.renderEngine == VRAY_RENDER && vrayFlag)
						data->SetLink(VRAYMATERIAL_LUMINANCE_SHADER, shd);
					else
						data->SetLink(MATERIAL_LUMINANCE_SHADER, shd);
				}
			}
			if(lsD.lsmaskfile != "")
			{
				shd2 = BaseShader::Alloc(Xbitmap);
				if(shd2)
				{
					shddata = shd2->GetDataInstance();
					// load the bitmap if it exists
					if(GeFExist(presets[presetIndex].path + Filename(lsD.lsmaskfile), FALSE))
					{
						shddata->SetFilename(BITMAPSHADER_FILENAME, presets[presetIndex].path + Filename(lsD.lsmaskfile));
						shd2->Message(MSG_UPDATE);
						if(theOpts.renderEngine == VRAY_RENDER && vrayFlag)
						{
							data->SetLink(VRAYMATERIAL_TRANSP_SHADER, shd2);
							data->SetBool(VRAYMATERIAL_TRANSP_TEXINVERT, TRUE);
						}
						else
							data->SetLink(MATERIAL_ALPHA_SHADER, shd2);
					}
				}
			}
			// set the color channel
			if(theOpts.renderEngine == VRAY_RENDER && vrayFlag)
				data->SetVector(VRAYMATERIAL_COLOR_COLOR, lsD.lscolor);
			else
				data->SetVector(MATERIAL_COLOR_COLOR, lsD.lscolor);
			// set the illumination multiplier and other options for Vray
			if(theOpts.renderEngine == VRAY_RENDER && vrayFlag)
			{
				if(theOpts.vrayLSMode == VRAY_LS_MODE_DI)
				{
					data->SetBool(VRAYMATERIAL_LUMINANCE_DIRECTILLUM, TRUE);
					data->SetBool(VRAYMATERIAL_LUMINANCE_EMITBACK, FALSE);
					data->SetReal(VRAYMATERIAL_LUMINANCE_INTENSITY, (lsD.lsmulti * theOpts.vrayLSIntensMulti));
					data->SetBool(VRAYMATERIAL_LUMINANCE_SHDON, TRUE);
					data->SetLong(VRAYMATERIAL_LUMINANCE_SUBDIVS, 64);
				}
				else if(theOpts.vrayLSMode == VRAY_LS_MODE_LUM)
				{
					data->SetReal(VRAYMATERIAL_LUMINANCE_SHADER_MULTIPLIER, (lsD.lsmulti * theOpts.vrayLSIntensMulti));
					data->SetBool(VRAYMATERIAL_LUMINANCE_EMITBACK, FALSE);
				}
			}
			else
				data->SetReal(MATERIAL_GLOBALILLUM_GENERATE_STRENGTH, lsD.lsmulti);
			if(shd)
				lsMat->InsertShader(shd, NULL);
			if(shd2)
				lsMat->InsertShader(shd2, NULL);
			lsMat->Message(MSG_UPDATE);
		}
		else if(lstype == LS_FRAME)
		{
			// set the color channel
			if(theOpts.renderEngine == VRAY_RENDER && vrayFlag)
			{
				data->SetBool(VRAYMATERIAL_USE_SPECULAR1, TRUE);
				data->SetVector(VRAYMATERIAL_COLOR_COLOR, lsD.lsframecolor);
			}
			else
				data->SetVector(MATERIAL_COLOR_COLOR, lsD.lsframecolor);
			lsMat->Message(MSG_UPDATE);
		}
		doc->InsertMaterial(lsMat, NULL, FALSE);
		doc->AddUndo(UNDOTYPE_NEW, lsMat);
		lsMat->Update(TRUE, TRUE);

		texTag = TextureTag::Alloc();
		if(texTag)
		{
			texTag->SetMaterial(lsMat);
			// set projection to UVW
			param.SetLong(TEXTURETAG_PROJECTION_UVW);
			texTag->SetParameter(DescID(TEXTURETAG_PROJECTION), param, DESCFLAGS_SET_0);
			doc->AddUndo(UNDOTYPE_CHANGE, obj);
			obj->InsertTag(texTag);
			obj->Message(MSG_UPDATE);
		}
	}

}

void SIBLDialog::SetupGI(BaseDocument *doc)
{
	RenderData *rdata = NULL;
	BaseVideoPost *pvp = NULL;
	Bool foundGI = FALSE;

	rdata = doc->GetActiveRenderData();
	if(rdata)
	{
		// check to make sure GI isn't already in the active render data
		pvp = rdata->GetFirstVideoPost();
		while(pvp)
		{
			if(pvp->GetType() == VPglobalillumination)
			{
				foundGI = TRUE;
				break;
			}
			pvp = pvp->GetNext();
		}
		if(!foundGI)
		{
			pvp = BaseVideoPost::Alloc(VPglobalillumination);
			if(pvp)
			{
				rdata->InsertVideoPost(pvp, NULL);
				doc->AddUndo(UNDOTYPE_NEW, pvp);
				rdata->Message(MSG_UPDATE);
			}
		}
	}
	else
		GePrint("sIBL Loader: Error - could not obtain the active render data to set up GI.");
}

Bool SIBLDialog::CreateSun(BaseDocument *doc, BaseObject *parent, BaseObject *prev)
{
	BaseObject *sunLight;
	Bool success = FALSE;
	GeData param;
	Vector sunRotPos;
	Real theta;
	BaseTag *ltag = NULL;
	Real mult;
	
	sunLight = BaseObject::Alloc(Olight);
	if(sunLight)
	{
		sunLight->SetName("sIBL_Sun");
		param.SetLong(LIGHT_TYPE_DISTANT);		// infinite light
		sunLight->SetParameter(DescID(LIGHT_TYPE), param, DESCFLAGS_SET_0);
		param.SetLong(2);		// hard shadows
		sunLight->SetParameter(DescID(LIGHT_SHADOWTYPE), param, DESCFLAGS_SET_0);
		param.SetReal(presets[presetIndex].sunmulti);		// sun brightness
		sunLight->SetParameter(DescID(LIGHT_BRIGHTNESS), param, DESCFLAGS_SET_0);
		param.SetVector(presets[presetIndex].suncolor);
		sunLight->SetParameter(DescID(LIGHT_COLOR), param, DESCFLAGS_SET_0);
		doc->InsertObject(sunLight, parent, prev, FALSE);
		doc->AddUndo(UNDOTYPE_NEW, sunLight);
		// are we using Vray?
		if(theOpts.renderEngine == VRAY_RENDER && vrayFlag)
		{
			// add a Vray light tag, which will take all the above values from the C4D light
			ltag = BaseTag::Alloc(Vlighttag);
			if(ltag)
			{
				doc->AddUndo(UNDOTYPE_CHANGE, sunLight);
				sunLight->InsertTag(ltag);
				// turn on physical sun and make it invisible
				param.SetLong(1);
				ltag->SetParameter(DescID(VRAYLIGHTTAG_SUN_ON), param, DESCFLAGS_SET_0);
				param.SetLong(1);
				ltag->SetParameter(DescID(VRAYLIGHTTAG_SUN_INVISIBLE), param, DESCFLAGS_SET_0);
				if (theOpts.vrayUseSunIntensity)
				{
					ltag->GetParameter(DescID(VRAYLIGHTTAG_SUN_STDINTMUL), param, DESCFLAGS_GET_0);			// get standard camera intensity multiplier
					mult = param.GetReal();
					param.SetReal(presets[presetIndex].sunmulti * mult);									// multiply it by the sun intensity and save it
					ltag->SetParameter(DescID(VRAYLIGHTTAG_SUN_STDINTMUL), param, DESCFLAGS_SET_0);
					ltag->GetParameter(DescID(VRAYLIGHTTAG_SUN_PHYINTMUL), param, DESCFLAGS_GET_0);			// get physical camera intensity multiplier
					mult = param.GetReal();
					param.SetReal(presets[presetIndex].sunmulti * mult);									// multiply it by the sun intensity and save it
					ltag->SetParameter(DescID(VRAYLIGHTTAG_SUN_PHYINTMUL), param, DESCFLAGS_SET_0);
				}
				sunLight->Message(MSG_UPDATE);
			}
		}
		// now set its rotation
		sunRotPos.x = (((presets[presetIndex].sunu + 0.25) * pi2) * -1.0) + Rad(180.0);			// correction of *-1.0 and +180 degrees is to allow inversion of texture tag
		theta = (pi05 - (presets[presetIndex].sunv * pi)) * -1;
		sunRotPos.y = theta;
		sunRotPos.z = 0.0;
		doc->AddUndo(UNDOTYPE_CHANGE, sunLight);
		sunLight->SetAbsRot(sunRotPos);
		// and change to an arbitrary position for user convenience
		sunRotPos.x = 0.0;
		sunRotPos.y = 500.00 * Sin(theta) * -1.0;
		sunRotPos.z = 0.0;
		doc->AddUndo(UNDOTYPE_CHANGE, sunLight);
		sunLight->SetAbsPos(sunRotPos);
		sunLight->Message(MSG_UPDATE);
		success = TRUE;
	}

	return success;
}

Bool SIBLDialog::CreateCompTag(BaseDocument *doc, BaseObject* skyObj, LONG skyType)
{
	BaseTag *comp;
	GeData param;
	Bool success = FALSE;

	comp = BaseTag::Alloc(Tcompositing);
	if(comp)
	{
		switch(skyType)
		{
		case SIBL_SKY_BG:
			// turn off GI and reflection
			param.SetLong(0);
			comp->SetParameter(DescID(COMPOSITINGTAG_CASTSHADOW), param, DESCFLAGS_SET_0);
			comp->SetParameter(DescID(COMPOSITINGTAG_RECEIVESHADOW), param, DESCFLAGS_SET_0);
			comp->SetParameter(DescID(COMPOSITINGTAG_SEENBYGI), param, DESCFLAGS_SET_0);
			comp->SetParameter(DescID(COMPOSITINGTAG_SEENBYREFLECTION), param, DESCFLAGS_SET_0);
			// turn on camera, transparency, rays, refraction, and ao
			param.SetLong(1);
			comp->SetParameter(DescID(COMPOSITINGTAG_SEENBYCAMERA), param, DESCFLAGS_SET_0);
			comp->SetParameter(DescID(COMPOSITINGTAG_SEENBYTRANSPARENCY), param, DESCFLAGS_SET_0);
			comp->SetParameter(DescID(COMPOSITINGTAG_SEENBYRAYS), param, DESCFLAGS_SET_0);
			comp->SetParameter(DescID(COMPOSITINGTAG_SEENBYREFRACTION), param, DESCFLAGS_SET_0);
			comp->SetParameter(DescID(COMPOSITINGTAG_SEENBYAO), param, DESCFLAGS_SET_0);
			break;

		case SIBL_SKY_EN:
			// turn off camera, rays, transparency
			param.SetLong(0);
			comp->SetParameter(DescID(COMPOSITINGTAG_CASTSHADOW), param, DESCFLAGS_SET_0);
			comp->SetParameter(DescID(COMPOSITINGTAG_RECEIVESHADOW), param, DESCFLAGS_SET_0);
			comp->SetParameter(DescID(COMPOSITINGTAG_SEENBYCAMERA), param, DESCFLAGS_SET_0);
			comp->SetParameter(DescID(COMPOSITINGTAG_SEENBYRAYS), param, DESCFLAGS_SET_0);
			comp->SetParameter(DescID(COMPOSITINGTAG_SEENBYTRANSPARENCY), param, DESCFLAGS_SET_0);
			// turn on gi
			param.SetLong(1);
			comp->SetParameter(DescID(COMPOSITINGTAG_SEENBYGI), param, DESCFLAGS_SET_0);
			break;

		case SIBL_SKY_RE:
			// turn off camera, gi, transparency, refraction, ao
			param.SetLong(0);
			comp->SetParameter(DescID(COMPOSITINGTAG_CASTSHADOW), param, DESCFLAGS_SET_0);
			comp->SetParameter(DescID(COMPOSITINGTAG_RECEIVESHADOW), param, DESCFLAGS_SET_0);
			comp->SetParameter(DescID(COMPOSITINGTAG_SEENBYCAMERA), param, DESCFLAGS_SET_0);
			comp->SetParameter(DescID(COMPOSITINGTAG_SEENBYGI), param, DESCFLAGS_SET_0);
			comp->SetParameter(DescID(COMPOSITINGTAG_SEENBYTRANSPARENCY), param, DESCFLAGS_SET_0);
			comp->SetParameter(DescID(COMPOSITINGTAG_SEENBYREFRACTION), param, DESCFLAGS_SET_0);
			comp->SetParameter(DescID(COMPOSITINGTAG_SEENBYAO), param, DESCFLAGS_SET_0);
			// turn on rays, reflection
			param.SetLong(1);
			comp->SetParameter(DescID(COMPOSITINGTAG_SEENBYRAYS), param, DESCFLAGS_SET_0);
			comp->SetParameter(DescID(COMPOSITINGTAG_SEENBYREFLECTION), param, DESCFLAGS_SET_0);
			break;

		default:
			break;
		}
		doc->AddUndo(UNDOTYPE_CHANGE, skyObj);
		skyObj->InsertTag(comp);
		skyObj->Message(MSG_UPDATE);
		success = TRUE;
	}

	return success;
}

Bool SIBLDialog::ApplySkyMat(BaseDocument *doc, Material *mat, BaseObject *skyObj)
{
	TextureTag* texTag;
	Bool success = FALSE;
	GeData param;

	texTag = TextureTag::Alloc();
	if(texTag)
	{
		texTag->SetMaterial(mat);
		// set projection to spherical
		param.SetLong(TEXTURETAG_PROJECTION_SPHERICAL);
		texTag->SetParameter(DescID(TEXTURETAG_PROJECTION), param, DESCFLAGS_SET_0);
		// reverse the bitmap, otherwise text is mirrored
		param.SetReal(-1.0L);
		texTag->SetParameter(DescID(TEXTURETAG_TILESX), param, DESCFLAGS_SET_0);
		doc->AddUndo(UNDOTYPE_CHANGE, skyObj);
		skyObj->InsertTag(texTag);
		skyObj->Message(MSG_UPDATE);
		success = TRUE;
	}

	return success;
}

Material* SIBLDialog::CreateSkyMat(BaseDocument *doc, String texFile, String matName, LONG skyType)
{
	Material *skyMat = NULL;
	BaseContainer *data, *shddata;
	BaseShader *shd = NULL;
	GeData param;
	LONG matTexPreview;

	skyMat = Material::Alloc();
	if(skyMat)
	{
		skyMat->SetName(matName);
		// environment map into luminance channel, others into color channel
		if(skyType == SIBL_SKY_EN)
		{
			skyMat->SetChannelState(CHANNEL_COLOR, FALSE);
			skyMat->SetChannelState(CHANNEL_LUMINANCE, TRUE);
		}
		else
		{
			skyMat->SetChannelState(CHANNEL_COLOR, TRUE);
			skyMat->SetChannelState(CHANNEL_LUMINANCE, FALSE);
		}
		skyMat->SetChannelState(CHANNEL_SPECULAR, FALSE);
		// set texture map
		data = skyMat->GetDataInstance();
		shd = BaseShader::Alloc(Xbitmap);
		if(shd)
		{
			shddata = shd->GetDataInstance();
			// load the bitmap if it exists
			if(GeFExist(presets[presetIndex].path + Filename(texFile), FALSE))
				shddata->SetFilename(BITMAPSHADER_FILENAME, presets[presetIndex].path + Filename(texFile));
			if(skyType == SIBL_SKY_EN)
				shddata->SetReal(BITMAPSHADER_GAMMA, presets[presetIndex].evgamma);
			else if(skyType == SIBL_SKY_RE)
				shddata->SetReal(BITMAPSHADER_GAMMA, presets[presetIndex].refgamma);
			shd->Message(MSG_UPDATE);
			if(skyType == SIBL_SKY_EN)								// environment map goes into luminance channel, the others into the color channel
				data->SetLink(MATERIAL_LUMINANCE_SHADER, shd);
			else
				data->SetLink(MATERIAL_COLOR_SHADER, shd);
			if(skyType == SIBL_SKY_BG)
			{
				switch(theOpts.texPreview)
				{
				case 7000:
					matTexPreview = MATERIAL_PREVIEWSIZE_DEF;
					break;
				case 7001:
					matTexPreview = MATERIAL_PREVIEWSIZE_64;
					break;
				case 7002:
					matTexPreview = MATERIAL_PREVIEWSIZE_128;
					break;
				case 7003:
					matTexPreview = MATERIAL_PREVIEWSIZE_256;
					break;
				case 7004:
					matTexPreview = MATERIAL_PREVIEWSIZE_512;
					break;
				case 7005:
					matTexPreview = MATERIAL_PREVIEWSIZE_1024;
					break;
				case 7006:
					matTexPreview = MATERIAL_PREVIEWSIZE_2048;
					break;
				case 7007:
					matTexPreview = MATERIAL_PREVIEWSIZE_4096;
					break;
				case 7008:
					matTexPreview = MATERIAL_PREVIEWSIZE_8192;
					break;

				default:
					matTexPreview = MATERIAL_PREVIEWSIZE_DEF;
					break;
				}

				param.SetLong(matTexPreview);
				skyMat->SetParameter(DescID(MATERIAL_PREVIEWSIZE), param, DESCFLAGS_SET_0);
			}
			skyMat->InsertShader(shd, NULL);
			skyMat->Message(MSG_UPDATE);
			doc->InsertMaterial(skyMat, NULL, FALSE);
			doc->AddUndo(UNDOTYPE_NEW, skyMat);
			skyMat->Update(TRUE, TRUE);
		}
		else
		{
			Material::Free(skyMat);
		}
	}
	return skyMat;
}

Bool SIBLDialog::CreateVrayBGSky(BaseDocument *doc, BaseObject *parent, BaseObject *prev)
{
	BaseMaterial *bgSkyMat = NULL;
	BaseContainer *data, *shddata;
	BaseShader *shd = NULL;
	Bool success = FALSE;
	BaseObject *skyObj;
	TextureTag* texTag;
	GeData param;
	LONG matTexPreview;

	bgSkyMat = BaseMaterial::Alloc(ID_VRAYBRDF_MATERIAL);
	if(bgSkyMat)
	{
		bgSkyMat->SetName("sIBL_Vray_editorBG");
		data = bgSkyMat->GetDataInstance();
		shd = BaseShader::Alloc(Xbitmap);
		if(shd)
		{
			shddata = shd->GetDataInstance();
			if(GeFExist(presets[presetIndex].path + presets[presetIndex].bgfile, FALSE))
				shddata->SetFilename(BITMAPSHADER_FILENAME, presets[presetIndex].path + presets[presetIndex].bgfile);
			shd->Message(MSG_UPDATE);
			data->SetLink(VRAYMATERIAL_COLOR_SHADER, shd);
			// set the preview resolution
			switch(theOpts.texPreview)
			{
			case 7000:
				matTexPreview = VRAYMATERIAL_PREVIEWSIZE_DEF;
				break;
			case 7001:
				matTexPreview = VRAYMATERIAL_PREVIEWSIZE_64;
				break;
			case 7002:
				matTexPreview = VRAYMATERIAL_PREVIEWSIZE_128;
				break;
			case 7003:
				matTexPreview = VRAYMATERIAL_PREVIEWSIZE_256;
				break;
			case 7004:
				matTexPreview = VRAYMATERIAL_PREVIEWSIZE_512;
				break;
			case 7005:
				matTexPreview = VRAYMATERIAL_PREVIEWSIZE_1024;
				break;
			case 7006:
			case 7007:
			case 7008:
				matTexPreview = VRAYMATERIAL_PREVIEWSIZE_2048;
				break;

			default:
				matTexPreview = VRAYMATERIAL_PREVIEWSIZE_DEF;
				break;
			}
			param.SetLong(matTexPreview);
			bgSkyMat->SetParameter(DescID(VRAYMATERIAL_PREVIEWSIZE), param, DESCFLAGS_SET_0);
			bgSkyMat->InsertShader(shd);
			doc->InsertMaterial(bgSkyMat);
			doc->AddUndo(UNDOTYPE_NEW, bgSkyMat);
			bgSkyMat->Update(TRUE, TRUE);
			success = TRUE;
		}
		else
		{
			BaseMaterial::Free(bgSkyMat);
		}
	}
	
	// create a sky object if we were successful in creating a sky mat
	if(bgSkyMat)
	{
		skyObj = BaseObject::Alloc(Osky);
		if(skyObj)
		{
			skyObj->SetName("sIBL_Vray_editorBG");
			doc->InsertObject(skyObj, parent, prev, FALSE);
			doc->AddUndo(UNDOTYPE_NEW, skyObj);
			texTag = TextureTag::Alloc();
			if(texTag)
			{
				texTag->SetMaterial(bgSkyMat);
				// set projection to spherical
				param.SetLong(TEXTURETAG_PROJECTION_SPHERICAL);
				texTag->SetParameter(DescID(TEXTURETAG_PROJECTION), param, DESCFLAGS_SET_0);
				// reverse the bitmap, otherwise text is mirrored
				param.SetReal(-1.0L);
				texTag->SetParameter(DescID(TEXTURETAG_TILESX), param, DESCFLAGS_SET_0);
				doc->AddUndo(UNDOTYPE_CHANGE, skyObj);
				skyObj->InsertTag(texTag);
				doc->AddUndo(UNDOTYPE_CHANGE, skyObj);
				skyObj->SetRenderMode(MODE_OFF);
				skyObj->Message(MSG_UPDATE);
			}
			else
				success = FALSE;
		}
		else
			success = FALSE;
	}

	return success;
}

void SIBLDialog::ScanCollection(Bool redrawFlag)
{
	// browse the collections folder for sIBL sets
	BaseContainer lvbc, bitbc, msg;
	LONG counter = 0L;
	Bool success;
	Filename thumb;
	String status;
	PresetSort sortPresets;
	
	// clear the list view of any existing content
	ClearPresets();
	
	// get the total number of presets, either in the current collection or all collections
	if(theOpts.loadAll)
	{
		counter = GetPresetCount(FALSE, Filename(""));
	}
	else
	{
		counter = GetPresetCount(TRUE, siblCollecs[currentColl].fname);
	}

	// allocate memory for the required presets
	if(counter > 0)
	{
		presets = bNew presetData[counter];
		if(!presets)
		{
			GePrint("Failed to allocate memory for preset files.");
			return;
		}
		else
		{
			// load the preset files into memory here
			if(theOpts.loadAll)
				success = LoadPresets(FALSE, Filename(""), counter);
			else
				success = LoadPresets(TRUE, siblCollecs[currentColl].fname, counter);
			if(success)
			{
				// sort the presets alphanumerically
				sortPresets.Sort(presets, counter, sizeof(presetData));
				if(redrawFlag)
					ShowThumbs(counter, " preset(s) loaded");
			}
			else
			{
				GePrint("sIBL Loader: failed to load preset files.");
			}
		}
	}

	presetCount = counter;				// set the number of presets
	// switch to browser tab
	SetLong(IDC_TABS, IDC_TAB1);
}

void SIBLDialog::ShowThumbs(LONG counter, String stat)
{
	BaseContainer lvbc, bitbc, msg;
	BitmapButtonCustomGui *browser;
	Filename thumb;
	String status;
	LONG i;

	// set up the bitmap button container
	bitbc.SetLong(BITMAPBUTTON_BORDER, BORDER_OUT);
	bitbc.SetBool(BITMAPBUTTON_BUTTON, TRUE);
	// fill the list view here
	for(i = 0L; i < counter; i++)
	{
		// add to preset list
		lvbc.SetString('name', presets[i].name);
		lvPresets.SetItem(i, lvbc);
		// add the thumbnails
		bitbc.SetString(BITMAPBUTTON_TOOLTIP, presets[i].name);
		AddCustomGui(SIBL_GUI_BITBUTTON + i, 1000479, "", 0L, 128, 96, bitbc);
		browser = NULL;
		browser = (BitmapButtonCustomGui*)FindCustomGui(SIBL_GUI_BITBUTTON + i, 1000479);
		if(browser)
		{
			thumb = presets[i].path + Filename(presets[i].icoFile);
			if(GeFExist(thumb))
				browser->SetImage(thumb, FALSE);
		}
	}
	msg.SetId(BFM_SETSTATUSBAR);
	status = LongToString(counter) + stat;
	msg.SetString(BFM_STATUSBAR_TXT, status);
	SendMessage(IDC_SCROLLTHUMBS, msg);
	LayoutChanged(IDC_THUMBNAILS);
	
	// select the first item in the list view
	if(lvPresets.GetItemCount() > 0)
	{
		// select the first item in the list
		AutoAlloc<BaseSelect> sel;
		if(!sel)
			return;
		sel->DeselectAll();
		sel->Select(0);
		lvPresets.SetSelection(sel);
	}
	lvPresets.DataChanged();

}

void SIBLDialog::ClearPresets(void)
{
	LONG i;
	lightData *lData;

	// clear the list view of any existing content
	for (i = lvPresets.GetItemCount()-1; i>=0; i--)
	{
		lvPresets.RemoveItem(i);
	}
	// delete any memory used for light data
	for(i = 0; i< presetCount; i++)
	{
		lData = presets[i].lights;
		bDelete(lData);
	}

	presetCount = 0L;
	bDelete(presets);
	presetIndex = -1L;
	// clear the thumbnails
	LayoutFlushGroup(IDC_THUMBNAILS);
	// initialise the GUI
	SetupGUI();
}

Bool SIBLDialog::LoadPresets(Bool bSingle, Filename fname, LONG counter)
{
	LONG index = 0L, i;

	if(numColls == 0)
		return FALSE;						// if no collections, can't be any presets

	if(bSingle)							// if we are only going to load one collection
	{
		index = LoadIBLSet(fname, index);
	}
	else								// we count all the presets in all collections
	{
		for(i = 0; i < numColls; i++)
		{
			index += LoadIBLSet(siblCollecs[i].fname, index);
		}
	}
	return TRUE;

}

LONG SIBLDialog::LoadIBLSet(Filename fileToLoad, LONG index)
{
	LONG counter = 0L;

	// get the count of presets in this folder
	counter = GetPresetCount(TRUE, fileToLoad);
	// load the presets into memory, not loading more than 'counter' files to avoid crashes
	AutoAlloc<BrowseFiles> bf;
	AutoAlloc<BrowseFiles> bf2;
	Filename fullPath;
	LONG i = 0L;
	void *buffer = NULL;

	buffer = GeAlloc(25000L);
	if(bf && buffer)
	{
		bf->Init(fileToLoad, 0L);
		while(bf->GetNext() && i < counter)
		{
			if(bf->IsDir() && bf2)		// if it's a folder
			{
				bf2->Init(fileToLoad + bf->GetFilename(), 0L);
				while(bf2->GetNext())
				{
					if(bf2->GetFilename().CheckSuffix("ibl"))
					{
						// show the .ibl file name
						fullPath = Filename(fileToLoad) + bf->GetFilename();
						presets[index+i].path  = fullPath;
						presets[index+i].file = bf2->GetFilename();
						InitPreset(index+i);
						ReadFile(fullPath, bf2->GetFilename(), index+i, (CHAR*)buffer);
						i++;
						break;
					}
				}
			}				
		}
	}
	GeFree(buffer);
	return counter;		// return number of presets loaded
}

void SIBLDialog::InitPreset(LONG index)
{
	// initialise a preset to default values

	presets[index].bgfile = "";
	presets[index].evfile = "";
	presets[index].reffile = "";
	presets[index].geoDate = "";
	presets[index].geoTime = "";
	presets[index].geoLat = "";
	presets[index].geoLong = "";
	presets[index].addsun = FALSE;		// assume no sun by default
	// set sun color to white, in case the file doesn't have a sun color key
	presets[index].suncolor = Vector(1.0, 1.0, 1.0);
	// set sun multi to 100% by default
	presets[index].sunmulti = 1.0;
	presets[index].numlights = 0L;		// no extra lights by default
	presets[index].lights = NULL;
}


VLONG SIBLDialog::ReadFile(Filename fullPath, Filename file, LONG index, CHAR *buffer)
{
	// read an .IBL file into a a buffer and return the number of characters read
	Filename fileToOpen;
	VLONG numBytes = 0, i;
	String line, key, value;
	CHAR oneChar[2], v;
	IniRead ir;
	LONG pos, numLights = 0L, numLS = 0L;

	AutoAlloc<BaseFile> bf;
	if(!bf)
	{
		GePrint("Failed to allocate BaseFile object.");
		return 0L;
	}

	fileToOpen = fullPath + file;
	if(bf->Open(fileToOpen, FILEOPEN_READ, FILEDIALOG_ANY))
	{
		numBytes = bf->ReadBytes(buffer, 25000, TRUE);
		bf->Close();
		// process the buffer
		line = "";
		for(i = 0L; i < numBytes; i++)
		{
			v = buffer[i];
			if(v == '\n' || v == '\r')		// end of line
			{
				// parse the line
				line = ir.leftTrim(line);
				// do we have a Lightsmith section?
				if(line.FindFirst("[Lightsmith", &pos, 0L))
					numLS++;
				// is this a light?
				else if(line.FindFirst("[Light", &pos, 0L))
					numLights++;

				if(line != "" && line[0] != ';' && line [0] != '[')		// can ignore these
				{
					key = ir.getKey(line);
					if(key != "")										// there is a key 
					{
						value = ir.getValue(line);
						SetKey(key, value, index);
					}
				}
				line = "";
			}
			else
			{
				oneChar[0] = v;
				oneChar[1] = '\0';
				line += oneChar;
			}
		}
	}

	// process any Lightsmiths
	presets[index].numLS = numLS;
	if(numLS > 0)
		ReadLS(numBytes, index, buffer);

	// set the number of additional lights
	presets[index].numlights = numLights;
	if(numLights > 0)
		ReadLights(numBytes, index, buffer);

	if(theOpts.correctSun)
		CorrectSunlight(index);

	return numBytes;
}

void SIBLDialog::CorrectSunlight(LONG index)
{
	Bool foundSun = FALSE;
	LONG i, j, first = 0;
	String lightStr;

	// if there is a sun in the file, then nothing to do
	if(presets[index].addsun)
		return;
	else		// there is no formal sun 
	{
		if(presets[index].numlights == 0)
			return;					// if there aren't any other lights, there can't be one called sun
		else
		{
			// check each light to see if it is called 'Sun'
			for(i = 0; i < presets[index].numlights; i++)
			{
				lightStr = presets[index].lights[i].lightname.ToLower();
				if(lightStr == "sun")
				{
					foundSun = TRUE;
					first = i;
					break;
				}
			}
			if(foundSun)			// there is a light called sun
			{
				// copy its parameters into the sun values of the preset
				presets[index].suncolor = presets[index].lights[first].lightcolor;
				presets[index].sunmulti = presets[index].lights[first].lightmulti;
				presets[index].sunu = presets[index].lights[first].lightu;
				presets[index].sunv = presets[index].lights[first].lightv;
				presets[index].addsun = TRUE;

				// now remove this light from the array of lights
				for(i = first, j = first + 1; j < presets[index].numlights; i++, j++)
				{
					presets[index].lights[i].lightname = presets[index].lights[j].lightname;
					presets[index].lights[i].lightcolor = presets[index].lights[j].lightcolor;
					presets[index].lights[i].lightmulti = presets[index].lights[j].lightmulti;
					presets[index].lights[i].lightu = presets[index].lights[j].lightu;
					presets[index].lights[i].lightv = presets[index].lights[j].lightv;
				}

				// decrement number of lights
				presets[index].numlights--;
			}
		}
	}


}

Bool SIBLDialog::ReadLights(VLONG numBytes, LONG index, CHAR* buffer)
{
	// read extra lights from the .ibl file stored in 'buffer'
	String line, key, value;
	String lightHeader;
	CHAR oneChar[2], v;
	IniRead ir;
	LONG numLights, lightIndex, i;
	Bool success = FALSE;
	LONG lightCount, pos;
	lightData *lightMem;

	// allocate memory for the lights
	lightCount = presets[index].numlights;
	lightMem = bNew lightData[lightCount];
	if(!lightMem)
	{
		GePrint("sIBL Loader: failed to allocate memory for light data.");
		return success;
	}
	else
	{
		presets[index].lights = lightMem;
		numLights = presets[index].numlights;
		lightIndex = 0;
		// process the buffer
		line = "";
		for(i = 0L; i < numBytes; i++)
		{
			v = buffer[i];
			if(v == '\n' || v == '\r')		// end of line
			{
				// parse the line
				line = ir.leftTrim(line);
				// is this a light header?
				if(line.FindFirst("[Light", &pos, 0L))
				{
					// get the light index
					lightHeader = line.SubStr(pos+6, line.GetLength() - pos);		// gets everything after '[Light'
					// remove the trailing ']'
					if(lightHeader[lightHeader.GetLength() - 1] == ']')
						lightHeader.Delete(lightHeader.GetLength() - 1, 1);
					// what's left should be a number
					lightIndex = lightHeader.ToLong();
				}
				else if(line != "" && line[0] != ';')		// can ignore these
				{
					key = ir.getKey(line);
					if(key != "")										// there is a key 
					{
						value = ir.getValue(line);
						SetLightKey(key, value, index, lightIndex-1);		// lightIndex - 1 because it will have values starting at 1
					}
				}
				line = "";
			}
			else
			{
				oneChar[0] = v;
				oneChar[1] = '\0';
				line += oneChar;
			}
		}
	}

	return success;
}

Bool SIBLDialog::ReadLS(VLONG numBytes, LONG index, CHAR* buffer)
{
	// read Lightsmiths from the .ibl file stored in 'buffer'
	String line, key, value;
	String lsheader;
	CHAR oneChar[2], v;
	IniRead ir;
	LONG numLS, lsIndex, i;
	Bool success = FALSE;
	LONG lscount, pos;
	lsData *lsMem;

	// allocate memory for the lightsmiths
	lscount = presets[index].numLS;
	lsMem = bNew lsData[lscount];
	if(!lsMem)
	{
		GePrint("sIBL Loader: failed to allocate memory for Lightsmith data.");
		return success;
	}
	else
	{
		presets[index].ls = lsMem;
		numLS = presets[index].numLS;
		lsIndex = 0;
		// process the buffer
		line = "";
		for(i = 0L; i < numBytes; i++)
		{
			v = buffer[i];
			if(v == '\n' || v == '\r')		// end of line
			{
				// parse the line
				line = ir.leftTrim(line);
				// is this a lightsmith header?
				if(line.FindFirst("[Lightsmith", &pos, 0L))
				{
					// get the lightsmith index
					lsheader = line.SubStr(pos+11, line.GetLength() - pos);		// gets everything after '[Lightsmith'
					// remove the trailing ']'
					if(lsheader[lsheader.GetLength() - 1] == ']')
						lsheader.Delete(lsheader.GetLength() - 1, 1);
					// what's left should be a number
					lsIndex = lsheader.ToLong();
				}
				else if(line != "" && line[0] != ';')		// can ignore these
				{
					key = ir.getKey(line);
					if(key != "")										// there is a key 
					{
						value = ir.getValue(line);
						SetLSKey(key, value, index, lsIndex-1);		// lsIndex - 1 because it will have values starting at 1
					}
				}
				line = "";
			}
			else
			{
				oneChar[0] = v;
				oneChar[1] = '\0';
				line += oneChar;
			}
		}
	}

	return success;
}

Bool SIBLDialog::ReloadFile()
{
	Bool success = FALSE;
	void *buffer = NULL;
	VLONG numBytes;

	buffer = GeAlloc(25000L);
	if(buffer)
	{
		numBytes = ReadFile(presets[presetIndex].path, presets[presetIndex].file, presetIndex, (CHAR *)buffer);
		if(numBytes > 0)
			success = TRUE;
	}
	ShowSetDetails(presetIndex);

	return success;
}

void SIBLDialog::SetKey(String key, String value, LONG index)
{
	// save the key-values into the presets data structure with index 'index'
	String keyLC = key.ToLower();		// in case it's a case-sensitive thing
	Real temp;
	LONG i;

	if(keyLC == "icofile")
	{
		presets[index].icoFile = value;
	}
	else if(keyLC == "previewfile")
	{
		presets[index].previewFile = value;
	}
	else if(keyLC == "name")
	{
		presets[index].name = value;
	}
	else if(keyLC == "author")
	{
		presets[index].author = value;
	}
	else if(keyLC == "location")
	{
		presets[index].location = value;
	}
	else if(keyLC == "comment")
	{
		presets[index].comment = value;
	}
	else if(keyLC == "geolat")
	{
		presets[index].geoLat = value;
	}
	else if(keyLC == "geolong")
	{
		presets[index].geoLong = value;
	}
	else if(keyLC == "link")
	{
		presets[index].link = value;
	}
	else if(keyLC == "date")
	{
		presets[index].geoDate = value;
	}
	else if(keyLC == "time")
	{
		presets[index].geoTime = value;
	}
	else if(keyLC == "height")
	{
		temp = value.ToReal();
		presets[index].height = temp;
	}
	else if(keyLC == "north")
	{
		temp = value.ToReal();
		presets[index].north = temp;
	}
	else if(keyLC == "bgfile")
	{
		presets[index].bgfile = value;
	}
	else if(keyLC == "bgmap")
	{
		i = value.ToLong();
		presets[index].bgmap = i;
	}
	else if(keyLC == "bgu")
	{
		temp = value.ToReal();
		presets[index].bgu = temp;
	}
	else if(keyLC == "bgv")
	{
		temp = value.ToReal();
		presets[index].bgv = temp;
	}
	else if(keyLC == "bgheight")
	{
		i = value.ToLong();
		presets[index].bgmap = i;
	}
	else if(keyLC == "evfile")
	{
		presets[index].evfile = value;
	}
	else if(keyLC == "evmap")
	{
		i = value.ToLong();
		presets[index].evmap = i;
	}
	else if(keyLC == "evu")
	{
		temp = value.ToReal();
		presets[index].evu = temp;
	}
	else if(keyLC == "evv")
	{
		temp = value.ToReal();
		presets[index].evv = temp;
	}
	else if(keyLC == "evheight")
	{
		i = value.ToLong();
		presets[index].evheight = i;
	}
	else if(keyLC == "evmulti")
	{
		temp = value.ToReal();
		presets[index].evmulti = temp;
	}
	else if(keyLC == "evgamma")
	{
		temp = value.ToReal();
		presets[index].evgamma = temp;
	}
	else if(keyLC == "reffile")
	{
		presets[index].reffile = value;
	}
	else if(keyLC == "refmap")
	{
		i = value.ToLong();
		presets[index].refmap = i;
	}
	else if(keyLC == "refu")
	{
		temp = value.ToReal();
		presets[index].refu = temp;
	}
	else if(keyLC == "refv")
	{
		temp = value.ToReal();
		presets[index].refv = temp;
	}
	else if(keyLC == "refheight")
	{
		i = value.ToLong();
		presets[index].refheight = i;
	}
	else if(keyLC == "refmulti")
	{
		temp = value.ToReal();
		presets[index].refmulti = temp;
	}
	else if(keyLC == "refgamma")
	{
		temp = value.ToReal();
		presets[index].refgamma = temp;
	}
	else if(keyLC == "suncolor")
	{
		presets[index].addsun = TRUE;		// we can add a sun if the user wants one
		presets[index].suncolor = GetColorFromValue(value);
	}
	else if(keyLC == "sunmulti")
	{
		temp = value.ToReal();
		presets[index].sunmulti = temp;
	}
	else if(keyLC == "sunu")
	{
		temp = value.ToReal();
		presets[index].sunu = temp;
	}
	else if(keyLC == "sunv")
	{
		temp = value.ToReal();
		presets[index].sunv = temp;
	}
}

void SIBLDialog::SetLightKey(String key, String value, LONG index, LONG lightIndex)
{
	// save the key-values into the presets data structure with index 'index' and light data structure with index 'lightIndex'
	String keyLC = key.ToLower();		// in case it's a case-sensitive thing
	Real temp;
	lightData *lData;

	lData = presets[index].lights;		// pointer to the lights data structure
	if(keyLC == "lightname")
	{
		lData[lightIndex].lightname = value;
	}
	else if(keyLC == "lightcolor")
	{
		lData[lightIndex].lightcolor = GetColorFromValue(value);
	}
	else if(keyLC == "lightmulti")
	{
		temp = value.ToReal();
		lData[lightIndex].lightmulti = temp;
	}
	else if(keyLC == "lightu")
	{
		temp = value.ToReal();
		lData[lightIndex].lightu = temp;
	}
	else if(keyLC == "lightv")
	{
		temp = value.ToReal();
		lData[lightIndex].lightv = temp;
	}
}

void SIBLDialog::SetLSKey(String key, String value, LONG index, LONG lsIndex)
{
	// save the key-values into the presets data structure with index 'index' and lightsmith data structure with index 'lsIndex'
	String keyLC = key.ToLower();		// in case it's a case-sensitive thing
	Real temp;
	lsData *lsD;
	LONG ltemp;

	lsD = presets[index].ls;		// pointer to the lights data structure
	if(keyLC == "lsname")
	{
		lsD[lsIndex].lsname = value;
	}
	else if(keyLC == "lsfile")
	{
		lsD[lsIndex].lsfile = value;
	}
	else if(keyLC == "lsmaskfile")
	{
		lsD[lsIndex].lsmaskfile = value;
	}
	else if(keyLC == "lswidth")
	{
		temp = value.ToReal();
		lsD[lsIndex].lswidth = temp;
	}
	else if(keyLC == "lsheight")
	{
		temp = value.ToReal();
		lsD[lsIndex].lsheight = temp;
	}
	else if(keyLC == "lsareawidth")
	{
		temp = value.ToReal();
		lsD[lsIndex].lsareawidth = temp;
	}
	else if(keyLC == "lsareaheight")
	{
		temp = value.ToReal();
		lsD[lsIndex].lsareaheight = temp;
	}
	else if(keyLC == "lscolor")
	{
		lsD[lsIndex].lscolor = GetColorFromValue(value);
	}
	else if(keyLC == "lsmulti")
	{
		temp = value.ToReal();
		lsD[lsIndex].lsmulti = temp;
	}
	if(keyLC == "lsobjectfile")
	{
		lsD[lsIndex].lsobjectfile = value;
	}
	else if(keyLC == "lsframefile")
	{
		lsD[lsIndex].lsframefile = value;
	}
	else if(keyLC == "lsframecolor")
	{
		lsD[lsIndex].lsframecolor = GetColorFromValue(value);
	}
	else if(keyLC == "lsscale")
	{
		temp = value.ToReal();
		lsD[lsIndex].lsscale = temp;
	}
	else if(keyLC == "lsshape")
	{
		ltemp = LS_SQUARE;
		if(value == "Square")
			ltemp = LS_SQUARE;
		else if(value == "Circular")
			ltemp = LS_CIRCULAR;
		lsD[lsIndex].lsshape = ltemp;
	}
}

Bool SIBLDialog::ReadLine(BaseFile *bf, String& line)
{
	CHAR oneChar[2];
	CHAR v;
	Bool success = TRUE;

	line = "";
	do
	{
		if(bf->ReadChar(&v))				// successfully got a char
		{
			if(v == '\n' || v == '\r')		// end of line
				break;
			else
			{
				oneChar[0] = v;
				oneChar[1] = '\0';
				line += oneChar;
			}
		}
		else								// must be at EOF
			success = FALSE;
	} while(success);

	return success;
}


LONG SIBLDialog::GetPresetCount(Bool bSingle, Filename fname)
{
	LONG iblcount = 0L, i;

	if(numColls == 0)
		return 0L;						// if no collections, can't be any presets

	if(bSingle)							// if we are only going to load one collection
	{
		iblcount = GetIBLCount(fname);	// count the .ibl files in the specified folder
	}
	else								// we count all the presets in all collections
	{
		for(i = 0; i < numColls; i++)
		{
			iblcount += GetIBLCount(siblCollecs[i].fname);
		}
	}
	return iblcount;
}

LONG SIBLDialog::GetIBLCount(Filename folderName)
{
	// browse the collections folder for sIBL sets
	AutoAlloc<BrowseFiles> bf;
	AutoAlloc<BrowseFiles> bf2;
	LONG i = 0L;

	if(bf)
	{
		bf->Init(folderName, 0L);
		while(bf->GetNext())
		{
			if(bf->IsDir() && bf2)		// if it's a folder
			{
				bf2->Init(folderName + bf->GetFilename(), 0L);
				while(bf2->GetNext())
				{
					if(bf2->GetFilename().CheckSuffix("ibl"))
					{
						// count the number of .ibl files
						i++;
						break;
					}
				}
			}				
		}
	}
	return i;
}

void SIBLDialog::ShowSetDetails(LONG index)
{
	Filename thumb;
	String linkText, labelText;
	LONG r, g, b;

	if(bbcg)
	{
		thumb = presets[index].path + Filename(presets[index].icoFile);
		if(GeFExist(thumb))
			bbcg->SetImage(thumb, FALSE);
	}
	if(hlcg)
	{
		linkText = presets[index].link;
		labelText = presets[index].link;
		hlcg->SetLinkString(&linkText, &labelText);
	}
	// is there a preview file?
	if(presets[index].previewFile.Content())
		Enable(IDC_SHOWPREVIEW, TRUE);
	else
		Enable(IDC_SHOWPREVIEW, FALSE);
	// set name
	SetString(IDC_PRESET_NAME, presets[index].name);
	// set author
	SetString(IDC_PRESET_AUTHOR, presets[index].author);
	// set location
	SetString(IDC_PRESET_LOCATION, presets[index].location);	
	// set comment
	comfieldUA.txtComment = presets[index].comment;
	comfieldUA.Redraw();
	// set latitude and longtitude
	SetString(IDC_LATITUDE, GetLatLong(presets[index].geoLat, TRUE));
	SetString(IDC_LONGTITUDE, GetLatLong(presets[index].geoLong, FALSE));
	// set date and time
	SetString(IDC_DATE, GetDateFromValue(presets[index].geoDate));
	SetString(IDC_TIME, GetTimeFromValue(presets[index].geoTime));
	SetString(IDC_LIGHTCOUNT, LongToString(presets[index].numlights));
	SetString(IDC_LSCOUNT, LongToString(presets[index].numLS));
	// draw sun color
	if(presets[index].addsun)		// there is a sun
	{
		suncolorUA.color = presets[index].suncolor;
	}
	else
	{
		// set sun to background color
		GetColorRGB(COLOR_BG, r, b, g);
		suncolorUA.color = Vector((Real)r/255, (Real)b/255, (Real)g/255);
	}
	suncolorUA.Redraw();
}

String SIBLDialog::GetLatLong(String degrees, Bool isLat)
{
	String txtDegrees;
	String txtQuadrant;
	GeData degData;
	Real rads;

	// if there is no lat/long data, return empty string
	if(degrees == "")
		return "";

	rads = Rad(degrees.ToReal());
	if(isLat)
	{
		if(rads<0)
		{
			rads *= -1.0;
			txtQuadrant = " S";
		}
		else
			txtQuadrant = " N";
	}
	else
	{
		if(rads<0)
		{
			rads *= -1.0;
			txtQuadrant = " W";
		}
		else
			txtQuadrant = " E";
	}
	degData.SetReal(rads);
	txtDegrees = FormatNumber(degData, FORMAT_DEGREE, 0L, TRUE);
	txtDegrees += txtQuadrant;

	return txtDegrees;
}


Vector SIBLDialog::GetColorFromValue(String value)
{
	// return a color as a vector from the string value passed
	// assume that the correct format = '255,255,255'

	LONG pos, start;
	String sub;
	Real r = 1.0, g = 1.0, b = 1.0;				// sun will be white if no values are found
	Vector color;
	IniRead ir;

	value = ir.allTrim(value);					// remove all leading and trailing spaces
	start = 0L;
	pos = 0L;
	if(value.FindFirst(",", &pos, start))		// get first comma position
	{
		sub = value.SubStr(0L, pos);
		r = sub.ToReal();
		r /= 255;								// convert to range 0.0 - 1.0
	}
	start = pos+1;
	if(value.FindFirst(",", &pos, start))		// second comma
	{
		sub = value.SubStr(start, pos-start);
		g = sub.ToReal();
		g /= 255;
	}
	start = pos+1;
	sub = value.SubStr(start, value.GetLength()-start);
	b = sub.ToReal();
	b /= 255;

	color = Vector(r, g, b);

	return color;
}

String SIBLDialog::GetTimeFromValue(String aTime)
{
	// takes a string in the EXIF time format and returns a formatted time, or a null string if failure
	String retTime;
	String hour = "", minute = "", second = "", hr12;
	LONG pos, start, temp;

	if(aTime == "")
		return "";

	// find first colon
	start = 0L;
	if(aTime.FindFirst(":", &pos, start))
	{
		hour = aTime.SubStr(start, pos);
	}
	// find second colon
	start = pos+1;
	if(start < aTime.GetLength())
	{
		if(aTime.FindFirst(":", &pos, start))
		{
			minute = aTime.SubStr(start, pos-start);
		}
	}
	start = pos+1;
	if(start < aTime.GetLength())
	{
		second = aTime.SubStr(start, aTime.GetLength()-start);
	}

	// get AM or PM
	temp = (hour.ToLong() * 3600) + (minute.ToLong() * 60) + second.ToLong();
	if(temp < SECONDS_PER_HALF_DAY)
	{
		hr12 = "AM";
	}
	else
	{
		hr12 = "PM";
		temp = hour.ToLong() - 12;
		hour = LongToString(temp);
	}
	if(hour.SubStr(0L, 1) == "0" && hour.GetLength() > 1)
		hour = hour.SubStr(1, 1);

	retTime = hour + ":" + minute + " " + hr12;

	return retTime;
}

String SIBLDialog::GetDateFromValue(String aDate)
{
	// takes a string in the EXIF date format and returns a formatted date, or a null string if failure
	String retDate;
	String year = "", month = "", day = "";
	LONG pos, start, temp;
	String monthNames[] = {
		String("January"),
		String("February"),
		String("March"),
		String("April"),
		String("May"),
		String("June"),
		String("July"),
		String("August"),
		String("September"),
		String("October"),
		String("November"),
		String("December")
	};

	if(aDate == "")
		return "";

	// find first colon
	start = 0L;
	if(aDate.FindFirst(":", &pos, start))
	{
		year = aDate.SubStr(start, pos);
	}
	// find second colon
	start = pos+1;
	if(start < aDate.GetLength())
	{
		if(aDate.FindFirst(":", &pos, start))
		{
			month = aDate.SubStr(start, pos-start);
		}
	}
	start = pos+1;
	if(start < aDate.GetLength())
	{
		day = aDate.SubStr(start, aDate.GetLength()-start);
	}

	// check the day
	if(day != "")
	{
		temp = day.ToLong();
		if(temp < 1 || temp > 31)
		{
			day = "";
		}
		else
		{
			if(day.SubStr(0L, 1) == "0" && day.GetLength() > 1)
				day = day.SubStr(1, 1);
			day += ", ";
		}
	}

	// get the month name
	if(month != "")
	{
		temp = month.ToLong();
		if(temp >= 1 && temp <= 12)
			month = monthNames[temp-1] + " ";
		else
			month = "";
	}

	retDate = month + day + year;
	
	return retDate;
}

Bool SIBLDialog::SavePrefs(void)
{
	// save the plugin preferences
	BaseContainer bc;
	LONG i;

	GetOpts();		// get the options from the dialog
	// save the number of collections folders
	bc.SetLong(SIBL_PREFS_MAXCOLLECTIONS, theOpts.maxCollecs);
	bc.SetLong(SIBL_PREFS_NUMCOLLECTIONS, numColls);
	// save the collection folders themselves
	if(numColls > 0)			// assumes there are some collection folders given
	{
		for(i = 0; i < numColls*2; i++)
		{
			bc.SetFilename(SIBL_PREFS_COLLECTPATH + (i*2), siblCollecs[i].fname);		// save the filename
			bc.SetString(SIBL_PREFS_COLLECTPATH + ((i*2) + 1), siblCollecs[i].name);			// save the name string
		}
	}
	// now the sIBL-Edit folder
	bc.SetFilename(SIBL_PREFS_SIBLEDITPATH, sibleditFolder);
	// the use spheres bool and radius
	bc.SetBool(SIBL_PREFS_USESPHERES, theOpts.useSpheres);
	bc.SetReal(SIBL_PREFS_SPHERERADIUS, theOpts.sphereRadius);
	// show vray editor background warning
	bc.SetBool(SIBL_PREFS_VRAYBGWARNING, theOpts.vrayWarnBG);
	// texture preview size
	bc.SetLong(SIBL_PREFS_TEXPREVIEWSIZE, theOpts.texPreview);
	// auto save prefs bool
	bc.SetBool(SIBL_PREFS_AUTOSAVEPREFS, theOpts.autoSavePrefs);
	// other preferences
	bc.SetBool(SIBL_PREFS_CORRECTSUN, theOpts.correctSun);
	bc.SetBool(SIBL_PREFS_WARNNOMATCHES, theOpts.warnNoMatches);
	bc.SetBool(SIBL_PREFS_WARNONCLEAR, theOpts.warnOnClear);
	// set the current collection as the default if desired
	if(theOpts.saveDefCollect)
		bc.SetLong(SIBL_PREFS_CURRENTCOLLECTION, currentColl);
	bc.SetReal(SIBL_PREFS_LS_SCALEMULTI, theOpts.ls_scalemulti);
	// save the preferences
	if(!SetWorldPluginData(ID_SIBLLOADER, bc, FALSE))
	{
		MessageDialog("sIBL Loader: failed to save preferences correctly.");
		return FALSE;
	}
	else
	{
		return TRUE;
	}
}

Bool SIBLDialog::SaveCollectionsPrefs(void)
{
	BaseContainer bc;
	LONG i;

	// save the number of collections and the collection paths to the preferences, merging with existing values
	bc.SetLong(SIBL_PREFS_NUMCOLLECTIONS, numColls);
	if(numColls > 0)			// assumes there are some collection folders given
	{
		for(i = 0; i < numColls*2; i++)
		{
			bc.SetFilename(SIBL_PREFS_COLLECTPATH + (i*2), siblCollecs[i].fname);		// save the filename
			bc.SetString(SIBL_PREFS_COLLECTPATH + ((i*2) + 1), siblCollecs[i].name);			// save the name string
		}
	}
	// now the sIBL-Edit folder
	bc.SetFilename(SIBL_PREFS_SIBLEDITPATH, sibleditFolder);
	if(SetWorldPluginData(ID_SIBLLOADER, bc, TRUE))
		return TRUE;
	else
		return FALSE;
}

Bool SIBLDialog::LoadPrefs(void)
{
	// load global plugin preferences
	BaseContainer *bc = NULL;
	GeData data;
	LONG i;

	bc = GetWorldPluginData(ID_SIBLLOADER);
	if(!bc)						// couldn't find any prefs, so use defaults
	{
		siblCollecs = NULL;
		siblCollecs = bNew collData[DEFAULT_MAX_COLLECTIONS];
		if(!siblCollecs)
		{
			GePrint("sIBL Loader: failed to allocate memory for collections.");
			return FALSE;
		}
		return FALSE;
	}

	// get collection values
	theOpts.maxCollecs = bc->GetLong(SIBL_PREFS_MAXCOLLECTIONS, DEFAULT_MAX_COLLECTIONS);
	numColls = bc->GetLong(SIBL_PREFS_NUMCOLLECTIONS, 0L);
	// allocate memory for the desired number of collections
	// use the larger of max number and actual number
	siblCollecs = NULL;
	i = (theOpts.maxCollecs >= numColls) ? theOpts.maxCollecs : numColls;
	siblCollecs = bNew collData[i];
	if(!siblCollecs)
	{
		GePrint("sIBL Loader: failed to allocate memory for collections.");
		return FALSE;
	}

	// get the collections themselves
	if(numColls > 0)
	{
		for(i = 0; i < numColls*2; i++)
		{
			siblCollecs[i].fname = bc->GetFilename(SIBL_PREFS_COLLECTPATH + (i*2), "");
			siblCollecs[i].name = bc->GetString(SIBL_PREFS_COLLECTPATH + ((i*2) + 1), "");
		}
		// get the desired default collection
		currentColl = bc->GetLong(SIBL_PREFS_CURRENTCOLLECTION, 0);
		// if we loaded collections, then set the first collection to be the active one if the current collection is out of range
		if(currentColl < 0 || currentColl >= numColls)
			currentColl = 0L;
	}
	else
		currentColl = -1L;				// if there are no collections, set the current one to -1 to stop the loader trying to show one

	// get sIBL-Edit path
	sibleditFolder = bc->GetFilename(SIBL_PREFS_SIBLEDITPATH, "");
	// get use spheres bool and radius
	theOpts.useSpheres = bc->GetBool(SIBL_PREFS_USESPHERES, FALSE);
	theOpts.sphereRadius = bc->GetReal(SIBL_PREFS_SPHERERADIUS, 5000.0);
	// texture preview size
	theOpts.texPreview = bc->GetLong(SIBL_PREFS_TEXPREVIEWSIZE, 7000);
	if(theOpts.texPreview < 7000)
		theOpts.texPreview = 7000;
	// autosave prefs bool
	theOpts.autoSavePrefs = bc->GetBool(SIBL_PREFS_AUTOSAVEPREFS, FALSE);
	// correct sun option
	theOpts.correctSun = bc->GetBool(SIBL_PREFS_CORRECTSUN, TRUE);
	theOpts.warnNoMatches = bc->GetBool(SIBL_PREFS_WARNNOMATCHES, FALSE);
	// show warning vray background editor
	theOpts.vrayWarnBG = bc->GetBool(SIBL_PREFS_VRAYBGWARNING, TRUE);
	// other options
	theOpts.warnOnClear = bc->GetBool(SIBL_PREFS_WARNONCLEAR, TRUE);
	theOpts.ls_scalemulti = bc->GetReal(SIBL_PREFS_LS_SCALEMULTI, 100.0);

	// update the interface
	// set the path to sIBL-Edit
	data = GeData(sibleditFolder);
	if(fnibled)
		fnibled->SetData(data);
	// add collections to drop down menus
	FreeChildren(IDC_COLLECTIONS);
	FreeChildren(IDC_CURRENTCOLL);
	for(i = 0; i < numColls; i++)
	{
		AddChild(IDC_COLLECTIONS, i, siblCollecs[i].name);
		AddChild(IDC_CURRENTCOLL, i, siblCollecs[i].name);
	}
	// set the menu to the current collection
	if(currentColl >= 0L)
	{
		SetLong(IDC_COLLECTIONS, currentColl);
		SetLong(IDC_CURRENTCOLL, currentColl);
	}
	SetLong(IDC_MAXCOLLECTIONS, theOpts.maxCollecs, 1, MAXLONGl);
	SetBool(IDC_USESPHERES, theOpts.useSpheres);
	SetMeter(IDC_DOMERADIUS, theOpts.sphereRadius, 0, MAXREALr);
	if(theOpts.useSpheres)
		Enable(IDC_DOMERADIUS, TRUE);
	else
		Enable(IDC_DOMERADIUS, FALSE);
	SetLong(IDC_TEXPREVIEW, theOpts.texPreview);
	SetBool(IDC_AUTOSAVEPREFS, theOpts.autoSavePrefs);
	SetBool(IDC_WARNNOMATCHES, theOpts.warnNoMatches);
	SetBool(IDC_CORRECTSUN, theOpts.correctSun);
	SetBool(IDC_VRAYBGWARNING, theOpts.vrayWarnBG);
	SetBool(IDC_WARNONCLEAR, theOpts.warnOnClear);
	SetBool(IDC_SETDEFAULTCOLL, theOpts.saveDefCollect);

	return TRUE;
}

// ------------------- end of dialog functions -----------------------------


// ------------------- start of Banner class functions----------------------
// definitions for the user area object
Banner::Banner()
{
	// allocate bitmap
	bmp = NULL;
	bmp = BaseBitmap::Alloc();
}
Banner::~Banner()
{
	// free the bitmap
	BaseBitmap::Free(bmp);
}
Bool Banner::Init()
{
	Filename fname;
	IMAGERESULT result;

	fname = GeGetPluginPath() + Filename("res");
	bmpToLoad.SetDirectory(fname);
	bmpToLoad.SetFile("banner.tif");
	if(bmp)
	{
		result = bmp->Init(bmpToLoad);
		if(result != IMAGERESULT_OK)
		{
			GePrint("sIBL Loader: Error " + LongToString(result) + " in loading bitmap " + bmpToLoad.GetString());
		}
	}

	return TRUE;
}

void Banner::DrawMsg(LONG x1, LONG y1, LONG x2, LONG y2, const BaseContainer &msg)
{
	// draw the bitmap
	if(bmp)
	{
		LONG w = GetWidth();
		LONG h = GetHeight();
		LONG xStart = (bmp->GetBw() - w)/2;
		if (xStart < 0)
			xStart = 0;
		LONG yStart = (bmp->GetBh() - h)/2;
		if (yStart < 0)
			yStart = 0;
		DrawBitmap(bmp, 0, 0, w, h, xStart, yStart, w, h, BMP_NORMALSCALED);
	}
}

// ------------------- end of Banner class functions-------------------------


// ------------------- start of CommentField class functions ----------------
Bool CommentField::Init()
{
	txtComment = "";
	return TRUE;
}

void CommentField::DrawMsg(LONG x1, LONG y1, LONG x2, LONG y2, const BaseContainer &msg)
{
	LONG fontHeight, i, yDraw, uaWidth, uaHeight, drawFrom, pos;
	String txtToDraw;
	
	uaWidth = GetWidth();
	uaHeight = GetHeight();
	if(txtComment != "")
	{
		// draw the text
		txtToDraw = txtComment;
		fontHeight = DrawGetFontHeight();
		yDraw = 0L;
		drawFrom = 0L;
		DrawSetPen(COLOR_BG);
		DrawRectangle(0L, 0L, uaWidth, uaHeight);
		DrawSetTextCol(COLOR_TEXT, COLOR_BG);
		while (yDraw < uaHeight && drawFrom < txtToDraw.GetLength())
		{
			for(i = 0; i < txtToDraw.GetLength(); i++)
			{
				if(DrawGetTextWidth(txtToDraw.SubStr(drawFrom, i+1)) > uaWidth)
				{
					if(txtToDraw.SubStr(drawFrom, i+1).FindLast(" ", &pos, i))
					{
						DrawText(txtToDraw.SubStr(drawFrom, pos+1), 0L, yDraw);
						drawFrom = drawFrom + pos + 1;
						break;
					}
					else		// we didn't find a space, so just truncate at the correct point
					{
						DrawText(txtToDraw.SubStr(drawFrom, i), 0L, yDraw);
						drawFrom = drawFrom + i;
						break;
					}
				}
				else if(i == txtToDraw.GetLength()-1)
				//else
				{
					// it's a short line, just draw it
					DrawText(txtToDraw.SubStr(drawFrom, i+1), 0L, yDraw);
					drawFrom = drawFrom + i +1;
					break;
				}
			}
			yDraw += fontHeight;		// move down to next line
		}		
	}
	else
	{
		DrawSetPen(COLOR_BG);
		DrawRectangle(0L, 0L, uaWidth, uaHeight);
	}
}


// ------------------- end of CommentField class functions ------------------

// ------------------- start of SunColorField functions
Bool SunColorField::Init()
{
	LONG r, g, b;

	// set colorfield to background
	GetColorRGB(COLOR_BG, r, b, g);
	color = Vector((Real)r/255, (Real)b/255, (Real)g/255);
	return TRUE;
}

void SunColorField::DrawMsg(LONG x1, LONG y1, LONG x2, LONG y2, const BaseContainer &msg)
{
	LONG uaWidth, uaHeight;
	
	uaWidth = GetWidth();
	uaHeight = GetHeight();
	DrawSetPen(color);
	DrawRectangle(0L, 0L, uaWidth, uaHeight);
}

// ------------------- end of SunColorField functions


// ------------------- start of PresetSort functions ------------------------

LONG PresetSort::Compare(void *a, void *b)
{
	presetData *a2, *b2;

	a2 = static_cast<presetData*>(a);
	b2 = static_cast<presetData*>(b);

	if(a2->name < b2->name)
		return -1;
	else if(a2->name == b2->name)
		return 0;
	else
		return 1;
}

// ------------------- end of PresetSort functions ---------------------------


// ------------------- start of plugin class functions ----------------------

LONG SIBLLoader::GetState(BaseDocument* doc)
{
	return CMD_ENABLED;
}

Bool SIBLLoader::Execute(BaseDocument *doc)
{
	return dlg.Open(DLG_TYPE_ASYNC, ID_SIBLLOADER);
}

// ------------------- end of plugin class functions ------------------------


// ------------------- start of global functions ----------------------------
// register the plugin
Bool RegisterSIBL(void)
{
	// decide by name if the plugin shall be registered - just for user convenience
	String sName("sIBL Loader");
	String sHelp("Loads and applies sIBL image sets");
	return RegisterCommandPlugin(ID_SIBLLOADER, sName, 0, AutoBitmap("siblloader.tif"), sHelp, gNew SIBLLoader);
}
// -------------------- end of global functions ----------------